/////////////////////////////////////////////////////////////////////////////
// XML Parser wrapper
//
// Written by Bjarke Viksoe (bjarke@viksoe.dk)
// Copyright (c) 2009 Bjarke Viksoe.
//
// This code may be used in compiled form in any way you desire. This
// source file may be redistributed by any means PROVIDING it is 
// not sold for profit without the authors written consent, and 
// providing that this notice and the authors name is included. 
//
// This file is provided "as is" with no expressed or implied warranty.
// The author accepts no liability if it causes any damage to you or your
// computer whatsoever. It's free, so don't hassle me about it.
//
// Beware of bugs.
//

#ifndef __XMLPARSER_XMLLITE_WRAPPER_H__
#define __XMLPARSER_XMLLITE_WRAPPER_H__

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

// Include Microsoft Xmllite library
#include <xmllite.h>
#pragma comment(lib, "xmllite.lib")

#include <string>


/////////////////////////////////////////////////////////////////////////
// Defines

class CXMLParser;

#ifndef _ASSERTE
   #define _ASSERTE(x)
#endif

#define XML_TEXT_TRUE "true"
#define XML_TEXT_FALSE "false"


/////////////////////////////////////////////////////////////////////////
// Templated Handler implementations

class CXMLElementHandler
{
public:
   virtual void OnParseInit(CXMLParser* /*pParser*/)
   {
   }

   virtual void OnStartElement(LPCWSTR /*pszName*/, LPCWSTR* /*ppszAttribs*/)
   {
   }

   virtual void OnEndElement(LPCWSTR /*pszName*/)
   {
   }

   virtual void OnCharacterData(LPCWSTR /*pszValue*/, INT /*cchValue*/)
   {
   }
};


/////////////////////////////////////////////////////////////////////////
// XML Parser wrapper

class CXMLParser
{
public:
   IXmlReader* m_pParser;
   CXMLElementHandler* m_pHandler;

   CXMLParser() : m_pParser(NULL), m_pHandler(NULL)
   {
   }

   ~CXMLParser()
   {
      Destroy();
   }

   operator IXmlReader*() const 
   { 
      return m_pParser;
   }

   bool Create()
   {
      _ASSERTE(m_pParser==NULL);
      ::CreateXmlReader(__uuidof(IXmlReader), reinterpret_cast<LPVOID*>(&m_pParser), 0);
      _ASSERTE(m_pParser);
      return m_pParser != NULL;
   }

   void Destroy()
   {
      if( m_pParser != NULL ) m_pParser->Release();
      m_pParser = NULL;
   }

   bool Parse(LPCVOID pXmlData, SIZE_T cbXml = (SIZE_T) -1)
   {
      _ASSERTE(m_pParser);
      if( cbXml == (SIZE_T) -1 ) cbXml = strlen( (const char*)pXmlData );
      IStream* pStream = NULL;
      if( ::CreateStreamOnHGlobal(0, TRUE, &pStream) != S_OK ) return false;
      DWORD dwBytesWritten = 0;
      pStream->Write(pXmlData, cbXml, &dwBytesWritten);
      if( cbXml != dwBytesWritten ) return false;
      LARGE_INTEGER ullSeekZero = { 0 };
      ULARGE_INTEGER ullNewPos = { 0 };
      pStream->Seek(ullSeekZero, STREAM_SEEK_SET, &ullNewPos);
      m_pParser->SetInput(pStream);
      LPCWSTR pszName, pszValue;
      XmlNodeType NodeType = XmlNodeType_None;
      while( m_pParser->Read(&NodeType) == S_OK )
      {
         switch( NodeType ) {
         case XmlNodeType_Element:
            if( m_pHandler != NULL ) {
               const int MAX_ATTTRIBS = 40;
               LPCWSTR aAttribs[(MAX_ATTTRIBS + 1) * 2];
               int nCount = 0;
               m_pParser->GetLocalName(&pszName, NULL);
               if( m_pParser->MoveToFirstAttribute() == S_OK ) {
                  for( ; nCount < MAX_ATTTRIBS * 2; ) {
                     if( !m_pParser->IsDefault() ) {
                        m_pParser->GetLocalName(&pszValue, NULL);
                        aAttribs[nCount++] = _wcsdup(pszValue);
                        m_pParser->GetValue(&pszValue, NULL);
                        aAttribs[nCount++] = _wcsdup(pszValue);
                     }
                     if( m_pParser->MoveToNextAttribute() != S_OK ) break;
                  }
               }
               aAttribs[nCount++] = NULL;
               aAttribs[nCount++] = NULL;
               m_pHandler->OnStartElement(pszName, aAttribs);
               for( int i = 0; i < nCount - 2; i++ ) free( (LPVOID)aAttribs[i] );
            }
            break;
         case XmlNodeType_EndElement:
            if( m_pHandler != NULL ) {
               m_pParser->GetLocalName(&pszName, NULL);
               m_pHandler->OnEndElement(pszName);
            }
            break;
         case XmlNodeType_Text:
         case XmlNodeType_Whitespace:
            if( m_pHandler != NULL ) {
               m_pParser->GetValue(&pszValue, NULL);
               m_pHandler->OnCharacterData(pszValue, wcslen(pszValue));
            }
            break;
         }
      }
      pStream->Release();
      return true;
   }

   void SetHandler(CXMLElementHandler* pHandler)
   {
      _ASSERTE(m_pParser);
      m_pHandler = pHandler;
      pHandler->OnParseInit(this);
   }

   void ClearHandler()
   {
      m_pHandler = NULL;
   }
};


/////////////////////////////////////////////////////////////////////////
// XML attribute map macros

#define BEGIN_XML_PARSE_MAP() std::wstring* _sRef; CXMLParser* _pParser; \
   void OnStartElement(LPCWSTR name, LPCWSTR* atts) { _sRef = NULL;

#define BEGIN_XML_ELEMENT(_name) if( wcscmp(L ## _name, name)==0 ) for( LPCWSTR* _atts = atts; *_atts; _atts += 2 ) {

#define XML_ELEMENT_TEST(_name, _expr) if( wcscmp(L ## _name, name)==0 ) { _expr; };

#define XML_ATTRIB_INIT(_expr) if( _atts == atts ) { _expr; };

#define XML_ATTRIB_STR(_attrib, _var) if( wcscmp(L ## _attrib, *_atts)==0 ) { _var = *(_atts+1); };

#define XML_ATTRIB_SZ(_attrib, _var) if( wcscmp(L ## _attrib, *_atts)==0 ) { wcsncpy(_var, *(_atts+1), (sizeof(_var)/sizeof(WCHAR))-1); _var[(sizeof(_var)/sizeof(WCHAR))-1] = '\0'; };

#define XML_ATTRIB_INT(_attrib, _var) if( wcscmp(L ## _attrib, *_atts)==0 ) { _var = _ttoi(*(_atts+1)); };

#define XML_ATTRIB_DBL(_attrib, _var) if( wcscmp(L ## _attrib, *_atts)==0 ) { _var = (double)_ttof(*(_atts+1)); };

#define XML_ATTRIB_BOOL(_attrib, _var) if( wcscmp(L ## _attrib, *_atts)==0 ) { _var = **(_atts+1)=='t' || **(_atts+1)=='T'; };

#define XML_ATTRIB_TEST(_attrib, _value, _expr) if( wcscmp(L ## _attrib, *_atts)==0 && wcscmp(_value, *(atts+1))==0 ) { _expr; };

#define XML_CHARDATA_STR(_name, _var) if( wcscmp(L ## _name, name)==0 ) { _sRef = &(_var); };

#define XML_IF(_expr) if( _expr ) {

#define XML_ENDIF() }

#define XML_CHANGE_HANDLER(_var) _pParser.SetHandler(_var);

#define END_XML_ELEMENT() }

#define END_XML_PARSE_MAP() }; \
   void OnCharacterData(LPCWSTR value, INT cch) { if( _sRef != NULL ) (*_sRef) += std::wstring(value, (size_t)cch); }; \
   void OnEndElement(LPCWSTR /*pszName*/) { _sRef = NULL; }; \
   void OnParserInit(CXMLParser* pParser) { _sRef = NULL; _pParser = pParser; };

#define XML_DEFINE_TEMP(_var) _var;


#endif // __XMLPARSER_XMLLITE_WRAPPER_H__
