/////////////////////////////////////////////////////////////////////////////
// XML Parser wrapper
//
// Written by Bjarke Viksoe (bjarke@viksoe.dk)
// Copyright (c) 2009 Bjarke Viksoe.
//
// This code may be used in compiled form in any way you desire. This
// source file may be redistributed by any means PROVIDING it is 
// not sold for profit without the authors written consent, and 
// providing that this notice and the authors name is included. 
//
// This file is provided "as is" with no expressed or implied warranty.
// The author accepts no liability if it causes any damage to you or your
// computer whatsoever. It's free, so don't hassle me about it.
//
// Beware of bugs.
//

#ifndef __XMLPARSER_EXPAT_WRAPPER_H__
#define __XMLPARSER_EXPAT_WRAPPER_H__

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

// Include open-source Expat library
#include "expat.h"
#pragma comment(lib, "libexpat.lib")

#include <string>


/////////////////////////////////////////////////////////////////////////
// Defines

#ifndef _ASSERTE
   #define _ASSERTE(x)
#endif

#define XML_TEXT_TRUE "true"
#define XML_TEXT_FALSE "false"


/////////////////////////////////////////////////////////////////////////
// Forward declares

class CXMLParser;


/////////////////////////////////////////////////////////////////////////
// Helper function

inline std::string utf8toLatin(const char* pstr) 
{
   // A map from the most-significant 6 bits of the first byte
   // the total number of bytes in a UTF-8 character.
   static char UTF8len[64] = 
   { 
      1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
      0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, /* erroneous */
      2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 5, 6 
   };

   size_t cch = strlen(pstr);

   char result[256];
   int s_pos = 0;
   int s_len = cch < 255 ? (int) strlen(pstr) : 255;
   int index = 0;

   while( s_pos < s_len ) {
     register unsigned int c = (int) pstr[s_pos];
     int len = UTF8len[ (c >> 2) & 0x3F ];
     s_pos++;

     register unsigned long u = 0;
     switch( len ) {
     case 6: u = c & 0x01; break;
     case 5: u = c & 0x03; break;
     case 4: u = c & 0x07; break;
     case 3: u = c & 0x0F; break;
     case 2: u = c & 0x1F; break;
     case 1: u = c & 0x7F; break;
     case 0: // Erroneous: c is the middle of a character.
       u = c & 0x3F; 
       len = 5; 
       break;
     }

     while( --len && (s_pos < s_len) ) {
        c = (int) pstr[s_pos];
        s_pos++;
        if( (c & 0xC0) == 0x80 ) {
          u = (u << 6) | (c & 0x3F);
        } 
        else { 
          // Unexpected start of a new character
          s_pos--;
          break;
        }
     }

     if( u <= 0xFF ) {
       // A valid character
       result[index++] = (char) u;
     }
     else { 
       // This character can't be represented in Latin-1
       result[index++] = '?';
     }
   }
   result[index] = '\0';

   return result;
}


/////////////////////////////////////////////////////////////////////////
// XML Handler implementations

class CXMLElementHandler
{
public:
   // Expat callbacks

   static void _OnStartElement(void* pUserData, const char* pszName, const char** ppszAttribs)
   {
      CXMLElementHandler* pThis = reinterpret_cast<CXMLElementHandler*>(pUserData);
      pThis->OnStartElement(pszName, ppszAttribs);
   }

   static void _OnEndElement(void* pUserData, const char* pszName)
   {
      CXMLElementHandler* pThis = reinterpret_cast<CXMLElementHandler*>(pUserData);
      pThis->OnEndElement(pszName);
   }

   static void _OnCharacterData(void* pUserData, const char* pszValue, int cchValue)
   {
      CXMLElementHandler* pThis = reinterpret_cast<CXMLElementHandler*>(pUserData);
      pThis->OnCharacterData(pszValue, cchValue);
   }

   // Default implementation

   virtual void OnParseInit(CXMLParser* /*pParser*/)
   {
   }

   virtual void OnStartElement(const char* /*pszName*/, const char** /*ppszAttribs*/)
   {
   }

   virtual void OnEndElement(const char* /*pszName*/)
   {
   }

   virtual void OnCharacterData(const char* /*pszValue*/, int /*cchValue*/)
   {
   }
};


/////////////////////////////////////////////////////////////////////////
// XML Parser wrapper

class CXMLParser
{
public:
   XML_Parser m_pParser;
   CXMLElementHandler* m_pHandler;

   CXMLParser() : m_pParser(NULL), m_pHandler(NULL)
   {
   }

   ~CXMLParser()
   {
      Destroy();
   }

   operator XML_Parser() const 
   { 
      return m_pParser;
   }

   bool Create()
   {
      Destroy();
      m_pParser = ::XML_ParserCreate(NULL);
      _ASSERTE(m_pParser);
      return m_pParser != NULL;
   }

   void Destroy()
   {
      if( m_pParser != NULL ) ::XML_ParserFree(m_pParser);
      m_pParser = NULL;
   }

   bool Reset()
   {
      _ASSERTE(m_pParser);
      return ::XML_ParserReset(m_pParser, NULL) == XML_TRUE; 
   }

   bool Parse(const char* pszXml, size_t cchXml = (size_t) -1, bool bIsFinal = true)
   {
      _ASSERTE(m_pParser);
      _ASSERTE(m_pHandler);
      if( cchXml == (size_t) -1 ) cchXml = strlen(pszXml);
      return ::XML_Parse(m_pParser, pszXml, (int) cchXml , bIsFinal) != XML_STATUS_ERROR;
   }

   void SetHandler(CXMLElementHandler* pHandler)
   {
      _ASSERTE(m_pParser);
      _ASSERTE(pHandler);
      m_pHandler = pHandler;
      pHandler->OnParseInit(this);
      ::XML_SetUserData(m_pParser, pHandler);
      ::XML_SetElementHandler(m_pParser, pHandler->_OnStartElement, pHandler->_OnEndElement);
      ::XML_SetCharacterDataHandler(m_pParser, pHandler->_OnCharacterData);
   }

   void ClearHandler()
   {
      m_pHandler = NULL;
      ::XML_SetElementHandler(m_pParser, NULL, NULL);
   }
};


/////////////////////////////////////////////////////////////////////////
// XML attribute map macros

#define BEGIN_XML_PARSE_MAP() std::string* _sRef; CXMLParser* _pParser; \
   void OnStartElement(const char* name, const char** atts) { _sRef = NULL;

#define BEGIN_XML_ELEMENT(_name) if( strcmp(name, _name)==0 ) for( const char** _atts = atts; *_atts; _atts += 2 ) {

#define XML_ELEMENT_TEST(_name, _expr) if( strcmp(_name, name)==0 ) { _expr; };

#define XML_ATTRIB_INIT(_expr) if( _atts == atts ) { _expr; };

#define XML_ATTRIB_STR(_attrib, _var) if( strcmp(_attrib, *_atts)==0 ) { _var = utf8toLatin(*(_atts+1)); };

#define XML_ATTRIB_SZ(_attrib, _var) if( strcmp(_attrib, *_atts)==0 ) { strncpy(_var, utf8toLatin(*(_atts+1)).c_str(), sizeof(_var)-1); _var[sizeof(_var)-1] = '\0'; };

#define XML_ATTRIB_INT(_attrib, _var) if( strcmp(_attrib, *_atts)==0 ) { _var = atoi(*(_atts+1)); };

#define XML_ATTRIB_DBL(_attrib, _var) if( strcmp(_attrib, *_atts)==0 ) { _var = (double)atof(*(_atts+1)); };

#define XML_ATTRIB_BOOL(_attrib, _var) if( strcmp(_attrib, *_atts)==0 ) { _var = **(_atts+1)=='t' || **(_atts+1)=='T'; };

#define XML_ATTRIB_TEST(_attrib, _value, _expr) if( strcmp(_attrib, *_atts)==0 && strcmp(_value, *(atts+1))==0 ) { _expr; };

#define XML_CHARDATA_STR(_name, _var) if( strcmp(_name, name)==0 ) { _sRef = &(_var); };

#define XML_IF(_expr) if( _expr ) {

#define XML_ENDIF() }

#define XML_CHANGE_HANDLER(_var) _pParser.SetHandler(_var);

#define END_XML_ELEMENT() }

#define END_XML_PARSE_MAP() }; \
   void OnCharacterData(const char* value, int cch) { if( _sRef != NULL ) (*_sRef) += std::string(value, (size_t)cch); }; \
   void OnEndElement(const char* /*pszName*/) { _sRef = NULL; }; \
   void OnParserInit(CXMLParser* pParser) { _sRef = NULL; _pParser = pParser; };

#define XML_DEFINE_TEMP(_var) _var;


#endif // __XMLPARSER_EXPAT_WRAPPER_H__

