// ThreadManager.cpp: implementation of the CThreadManager class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "WebPageLoader.h"
#include "ThreadManager.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction

CThreadManager::CThreadManager()
{
   m_nActiveThreads = m_nTotalThreads = m_nMaxThreads = 0;
}

CThreadManager::~CThreadManager()
{
   TRACE(_T("Running ThreadManager cleanup...\n"));
   
   CSingleLock lock(&m_cs, TRUE);
   
   // Remove all thread queues and stuff...
   while( !m_IdleQueue.IsEmpty() ) {
      CSession *pSession = m_IdleQueue.RemoveHead();
      ASSERT_VALID(pSession);
      TRACE(_T("Deleting session %ld.\n"), pSession->m_iUniqueID);
      pSession->Release();
   }
   while( !m_StoppedQueue.IsEmpty() ) {
      CSession *pSession = m_StoppedQueue.RemoveHead();
      ASSERT_VALID(pSession);
      TRACE(_T("Deleting session %ld.\n"), pSession->m_iUniqueID);
      pSession->Release();
   }
   while( !m_SessionQueue.IsEmpty() ) {
      CSession *pSession = m_SessionQueue.RemoveHead();
      ASSERT_VALID(pSession);
      TRACE(_T("Deleting session %ld.\n"), pSession->m_iUniqueID);
      pSession->Release();
   }
}


//////////////////////////////////////////////////////////////////////
// Master functions

void CThreadManager::Close()
{
   // Stop running threads
   StopRunningThreads();

   // Wait for running threads
   DWORD dwTick = ::GetTickCount();
   while( HasRunningThreads() ) {
      ::Sleep(200L);
      if( ::GetTickCount() - dwTick > 2000L ) 
         break;
   }
}

BOOL CThreadManager::ScheduleRun()
{
   BOOL bRefresh = FALSE;
   bRefresh |= MoveStoppedThreads();
   bRefresh |= ScheduleNewThreads();
   bRefresh |= CheckForKilledThreads();
   bRefresh |= RemoveKilledThreads();
   return bRefresh;
};

CSession* CThreadManager::FindSession(LONG ID, SessionQueueType Type /*= QUEUE_ALL */)
{
   CSession *session;
   POSITION pos;
   if( Type & QUEUE_WAIT ) {
      pos = m_IdleQueue.GetHeadPosition();
      while( pos != NULL ) {
         session = m_IdleQueue.GetNext(pos);
         ASSERT_VALID(session);
         if( session->m_iUniqueID == ID ) 
            return session;
      }
   }
   if( Type & QUEUE_RUNNING ) {
      pos = m_SessionQueue.GetHeadPosition();
      while( pos != NULL ) {
         session = m_SessionQueue.GetNext(pos);
         ASSERT_VALID(session);
         if( session->m_iUniqueID == ID ) 
            return session;
      }
   }
   if( Type & QUEUE_STOPPED ) {
      pos = m_StoppedQueue.GetHeadPosition();
      while( pos != NULL ) {
         session = m_StoppedQueue.GetNext(pos);
         ASSERT_VALID(session);
         if( session->m_iUniqueID == ID ) 
            return session;
      }
   }
   return NULL;
}

void CThreadManager::GetSessions(CSessionList& List, SessionQueueType Type /*= QUEUE_ALL */)
{
   List.RemoveAll();
   POSITION pos;
   if( Type & QUEUE_WAIT ) {
      pos = m_IdleQueue.GetHeadPosition();
      while( pos != NULL )
         List.AddTail( m_IdleQueue.GetNext(pos) );
   }
   if( Type & QUEUE_RUNNING ) {
      pos = m_SessionQueue.GetHeadPosition();
      while( pos != NULL )
         List.AddTail( m_SessionQueue.GetNext(pos) );
   }
   if( Type & QUEUE_STOPPED ) {
      pos = m_StoppedQueue.GetHeadPosition();
      while( pos != NULL )
         List.AddTail( m_StoppedQueue.GetNext(pos) );
   }
}


//////////////////////////////////////////////////////////////////////
// Management functions

void CThreadManager::AddSession(CSession* pSession)
{
   ASSERT_VALID(pSession);
   
   CSingleLock lock(&m_cs, TRUE);
   
   pSession->SetState(STATE_QUEUED);
   
   m_IdleQueue.AddTail( pSession );
   
   pSession->PostUpdate(WM_SCHEDULE);
}

void CThreadManager::ThreadSleep()
{
}

void CThreadManager::ThreadResume()
{
}

void CThreadManager::ThreadStop()
{
}

BOOL CThreadManager::MoveStoppedThreads()
{
   CSingleLock lock(&m_cs, TRUE);

   BOOL bProcessed = FALSE;

   // Move stopped threads to idle list...
   POSITION pos;
   for( pos = m_SessionQueue.GetHeadPosition(); pos != NULL; ) {
      POSITION oldpos = pos;
      
      CSession *pSession = m_SessionQueue.GetNext(pos);
      ASSERT_VALID(pSession);
      
      if( pSession->m_State == STATE_STOPPED ) 
      {
         CSessionDataLock lock(pSession);
         
         // Remove the session from the running queue
         m_SessionQueue.RemoveAt(oldpos);
                 
         // Now add session to idle queue...
         m_StoppedQueue.AddHead(pSession);
         
         // Statistics...
         m_nActiveThreads--;
         
         // Done
         bProcessed = TRUE;
         
         // Start from the head again...
         pos = m_SessionQueue.GetHeadPosition();      
      }
   }
   return bProcessed;
}

BOOL CThreadManager::CheckForKilledThreads()
{
   CSingleLock lock(&m_cs, TRUE);
   
   BOOL bProcessed = FALSE;
   
   // If any session has been marked as killed, we should
   // make sure it will be removed when the thread is done!
   CSessionList all;
   GetSessions(all);

   POSITION pos = all.GetHeadPosition();
   while( pos != NULL ) {
      CSession *pSession = all.GetNext(pos);
      
      bool bRemove = false;      
      switch( pSession->m_State ) {
      case STATE_STOPPED:
         if( pSession->m_bKillRequest ) 
            bRemove = true;
         if( pSession->m_pPreferences->m_bDeleteThreadWhenDone ) 
            bRemove = true;
         break;
      case STATE_QUEUED:
      case STATE_SLEEPING:
         if( pSession->m_bKillRequest ) 
            bRemove = true;
         break;
      case STATE_RUNNING:
      case STATE_CONNECTING:
         if( pSession->m_bKillRequest ) 
         {           
            ::Sleep(200);            
  
            CSessionDataLock lock(pSession);

            BOOL bOnline = FALSE;
            pSession->Abort(bOnline);

            if( bOnline )
               bProcessed = TRUE; // yes, refresh again
         }
         break;
      }

      if( bRemove ) 
      {
         ASSERT_VALID(pSession);
         
         CSessionDataLock lock(pSession);
         
         // Session will be marked dead!
         // Cleanup code will remove it from queue later.
         pSession->m_State = STATE_KILLED;         
         
         bProcessed = TRUE;
      }
   }

   return bProcessed;
};

BOOL CThreadManager::ScheduleNewThreads()
{
   CSingleLock lock(&m_cs, TRUE);

   BOOL bProcessed = FALSE;
   
   // If any vacant slots, we remove the first thread from the
   // session queue and schedule it...
   if( m_SessionQueue.GetCount() < m_nMaxThreads && m_IdleQueue.GetCount() > 0 ) 
   {
      CSession *pSession = m_IdleQueue.RemoveHead();
      ASSERT_VALID(pSession);

      CSessionDataLock lock(pSession);
      
      // Session is now connecting
      pSession->m_State = STATE_CONNECTING;

      // So add it to the running queue
      m_SessionQueue.AddTail(pSession);
      
      // Spawn download thread
      CWinThread *pThread = AfxBeginThread( DownloadSessionThread, pSession );

      pSession->m_Threads.Add( pThread->m_hThread );
      
      m_nActiveThreads++;
      m_nTotalThreads++;
      
      // Done for now..
      bProcessed = TRUE;
   }

   return bProcessed;
};

BOOL CThreadManager::RemoveKilledThreads()
{
   CSingleLock lock( &m_cs, TRUE );

   BOOL bProcessed = FALSE;

   // Move stopped threads to idle list...
   CSessionList all;
   GetSessions(all);

   POSITION pos = all.GetHeadPosition();
   while( pos != NULL ) {
      CSession *pSession = all.GetNext(pos);
      ASSERT_VALID(pSession);

      if( pSession->m_State == STATE_KILLED ) 
      {
         CWaitCursor cursor;

         CAutoRefCount ref(pSession);

         CSingleLock lock(&m_cs, TRUE);

         // First, remove it from ALL queues...
         POSITION delpos;
#define DeleteQueueEntry(queue,entry)  delpos = queue.Find(entry); if( delpos != NULL ) queue.RemoveAt(delpos)
         DeleteQueueEntry(m_StoppedQueue, pSession);
         DeleteQueueEntry(m_IdleQueue, pSession);
         DeleteQueueEntry(m_SessionQueue, pSession);
         DeleteQueueEntry(all, pSession);
#undef DeleteQueueEntry

         lock.Unlock();

         TRACE(_T("Checking for thread termination.\n"));

         BOOL bWasAlive = FALSE;
         for( int i = 0; i < pSession->m_Threads.GetSize(); i++ ) {
            DWORD dwCode = 0;
            if( ::GetExitCodeThread(pSession->m_Threads[i], &dwCode) == FALSE )
               if( dwCode == STILL_ACTIVE ) 
                  bWasAlive = TRUE;
         }
         if( bWasAlive ) {
            TRACE(_T("Threads not complete!\n"));
            ::Sleep(300);
         }

         pSession->Release();
         
         bProcessed = TRUE;

         // Restart search...
         pos = all.GetHeadPosition();
      }
   }

   return bProcessed;
}

BOOL CThreadManager::HasRunningThreads()
{
   CSingleLock lock(&m_cs, TRUE);

   CSessionList all;
   GetSessions(all);

   POSITION pos = all.GetHeadPosition();
   while( pos != NULL ) {
      CSession *pSession = all.GetNext(pos);
      switch( pSession->m_State ) {
      case STATE_RUNNING:
      case STATE_CONNECTING:
         return TRUE;
      }
   }

   return FALSE;
}

void CThreadManager::StopRunningThreads()
{
   {
      CSingleLock lock(&m_cs, TRUE);
      
      CSessionList all;
      GetSessions(all);
      
      POSITION pos = all.GetHeadPosition();
      while( pos != NULL ) {
         CSession *pSession = all.GetNext(pos);
         switch( pSession->m_State ) {
         case STATE_RUNNING:
         case STATE_CONNECTING:
            pSession->m_bKillRequest = true;
            break;
         }
      }
   }

   MoveStoppedThreads();
   
   CheckForKilledThreads();
   
   RemoveKilledThreads();
}

