// Session.cpp: implementation of the CSession class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "WebPageLoader.h"
#include "Session.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

IMPLEMENT_DYNCREATE(CSession, CWinThread)

volatile LONG s_iUniqueCount = 0;


CSession::CSession()
{
   m_State = m_OldState = STATE_UNSCHEDULED;
   m_pPreferences = NULL;
   m_nStartIndex = m_nStopIndex = 0;
   m_bSleepRequest = m_bStopRequest = m_bKillRequest = false;

   m_FilesHash.InitHashTable(797);

   m_pPreferences = NULL;
   m_pCurrentDownloadFile = NULL;

   // Here we assign a unique ID.
   // A cool way to do it, but InterlockedIncrement() returns
   // crap on NT3.51 / Win95
   m_iUniqueID = ::InterlockedIncrement(&s_iUniqueCount);
}

CSession::~CSession()
{
}


//////////////////////////////////////////////////////////////////////
// Implementation

void CSession::Create(SessionType Type, const CSessionSettings& Info, CPreferences* pPrefs)
{
   ASSERT(pPrefs);
   m_Type = Type;
   m_Settings = Info;
   m_pPreferences = pPrefs;
};

void CSession::Start()
{
   // Extract the server URL into components, and store them in SessionInfo struct
   DWORD dwType;
   INTERNET_PORT nPort;
   AfxParseURL( m_sURL, dwType, m_Info.m_sServer, m_Info.m_sPage, nPort );

   if( m_Info.m_sPage.Right(1) != _T('/') ) {
      int iPos = m_Info.m_sPage.ReverseFind(_T('/'));
      if( iPos > 0 )
         m_Info.m_sPage = m_Info.m_sPage.Left(iPos + 1);
   }

   m_Info.m_tStarted = CTime::GetCurrentTime();

   m_State = STATE_CONNECTING;

   PostUpdate(WM_SCHEDULE);
   PostUpdate(WM_REFRESHITEMS, m_iUniqueID);

   Log(LOGTYPE_LOG, IDS_LOG_SESSIONSTARTED, "");
};

void CSession::Done()
{
   m_Info.m_tStopped = CTime::GetCurrentTime();

   m_State = STATE_STOPPED;

   PostUpdate(WM_SCHEDULE);
   PostUpdate(WM_REFRESHITEMS, m_iUniqueID);

   Log(LOGTYPE_LOG, IDS_LOG_SESSIONDONE, "");
}

BOOL CSession::Abort(BOOL& bWasOnline)
{
   CDownloadFile* pFile = m_pCurrentDownloadFile;
   if( pFile != NULL ) pFile->Abort(bWasOnline);

   return TRUE;
}

BOOL CSession::ScheduleNewDownload()
// Using the updated class attributes, we schedule a
// new download session. If this is a Image Scan then
// we create file downloads for all files.
{
   if( m_Files.GetCount() > 0 ) 
      return FALSE;

   BOOL bRes = TRUE;

   CSessionDataLock  lock1(this);
   CSessionFilesLock lock2(this);
   CSessionHashLock  lock3(this);

   switch( m_Type ) {
   case TYPE_HTMLSCAN:
      {
         CDownloadFile *pFile = new CDownloadFile();
         pFile->Create(this, m_sURL, _T(""), FALSE);
         m_Files.AddHead(pFile);
         m_FilesHash.SetAt(m_sURL, 1);
      }
      break;
   case TYPE_IMAGESCAN:
      {
         // Set some default settings
         // (they are not displayed in the property sheets for image
         //  scans)
         m_Settings.m_Duplicates = DUP_OVERWRITE;
         m_Settings.m_bUseHtmlFileNameFilter = FALSE;
         m_Settings.m_bUseImageFileNameFilter = FALSE;
         m_Settings.m_bUseFileSizeFilter = FALSE;
         
         // Submit all the image files...
         for( int i = m_nStopIndex; i >= m_nStartIndex; i-- ) {
            CString sURL;
            sURL.Format(m_sFormat, i);
            
            CDownloadFile *pFile = new CDownloadFile();
            pFile->Create(this, sURL, _T(""), TRUE);
            m_Files.AddHead(pFile);
            m_FilesHash.SetAt(m_sURL, 1);
         }
      }
      break;
   default:      
      bRes = FALSE;  // Error: invalid type
      break;
   }

   return bRes;
}

void CSession::PostUpdate(UINT uMsg, WPARAM wParam, LPARAM lParam)
{
   CWnd* pMainFrm = AfxGetApp()->GetMainWnd();
   ASSERT(pMainFrm);

   if( pMainFrm != NULL ) 
      pMainFrm->PostMessage(uMsg, wParam, lParam);
}

void CSession::Log(LogType Type, UINT uMsgID, CString sPage)
// Add to the log.
{
   if( !BfxIsValidSession(this) ) return;
  
   CSessionLogLock lock(this);

   CTime t;
   t = CTime::GetCurrentTime();

   CLog log;
   log.Type = Type;
   log.tTime = t;
   log.uMsgID = uMsgID;
   log.sPage = sPage;
   m_Logs.Add(log);

   // Make sure to update bottom pane with new info
   PostUpdate(WM_REFRESHNODES, m_iUniqueID);
}

void CSession::SetState(SessionState State)
{
   m_State = State;

   PostUpdate(WM_SCHEDULE);
   PostUpdate(WM_REFRESHITEMS, m_iUniqueID);
}

CString CSession::GetBaseDomain() const
{
   // Extract the server part of the URL
   CString sServer, sPage;
   DWORD dwType;
   INTERNET_PORT nPort;
   AfxParseURL( m_sURL, dwType, sServer, sPage, nPort );
   int nDots = 0;
   for( int i = 0; i < sServer.GetLength(); i++ ) 
      if( sServer.GetAt(i) == '.' ) 
         ++nDots;
   while( nDots >= 2 ) {
      int iPos = sServer.Find('.');
      sServer = sServer.Mid(iPos + 1);
      nDots = 0;
      for( int i = 0; i < sServer.GetLength(); i++ ) 
         if( sServer.GetAt(i) == '.' ) 
            ++nDots;
   }
   return sServer;
}

BOOL CSession::RescheduleBrokenDownloads(int iMaxAttempts)
{
   CSessionFilesLock lock(this);

   POSITION pos = m_Files.GetHeadPosition();
   BOOL bFound = FALSE;
   while( pos!=NULL ) {
      CDownloadFile *pInetFile = m_Files.GetNext(pos);
      if( pInetFile->m_State == FILESTATE_BROKEN 
          && pInetFile->m_iDownloadAttempts < iMaxAttempts ) 
      {
         pInetFile->m_State = FILESTATE_WAITING;
         bFound = TRUE;
      }
   }
   return bFound;
}

CDownloadFile *CSession::GetNextDownload()
{
   CSessionFilesLock lock(this);
   
   POSITION pos = m_Files.GetHeadPosition();
   while( pos!=NULL ) {
      CDownloadFile *pInetFile = m_Files.GetNext(pos);

      if( pInetFile->m_State != FILESTATE_WAITING )
         continue;

      // Found one, let's continue to scan for priority
      // downloads as well...

      if( m_Settings.m_bUsePriorityFileNameFilter || m_Settings.m_bUseImmediateDownload ) 
      {
         while( pos != NULL ) {
            CDownloadFile *pInetTemp = m_Files.GetNext(pos);
            if( pInetTemp->m_State != FILESTATE_WAITING || pInetTemp->m_iPriority <= pInetFile->m_iPriority )
               continue;
            pInetFile = pInetTemp;
         }

         if( pInetFile->m_iPriority < 20 )
            m_Settings.m_bUseImmediateDownload = FALSE;
      }

      return pInetFile;
   }
   
   return NULL;
}

void CSession::RandomizeFiles(DWORD dwTimeout)
{
   CSessionFilesLock lock(this);

   DWORD dwTick = ::GetTickCount();
   
   int iStart = m_Info.m_nFilesDownloaded + m_Info.m_nFilesSkipped;
   if( m_Settings.m_bUsePriorityFileNameFilter )
      iStart = 1;

   int nCount = m_Files.GetCount() - iStart;
   while( nCount > 0 ) {
      int n = (::rand() % nCount) + iStart;
      POSITION pos = m_Files.FindIndex(n);
      while( pos != NULL ) {
         POSITION posOld = pos;
         CDownloadFile *pFile = m_Files.GetNext(pos);
         if( pos == NULL || pFile->m_State != FILESTATE_WAITING)
            continue;
         m_Files.RemoveAt(posOld);
         m_Files.AddTail(pFile);
         break;
      }
      
      DWORD dwNow = ::GetTickCount();
      if( (long)(dwNow - dwTick) < 0 ) 
         break;
      if( dwNow - dwTick > dwTimeout ) 
         break;
   }
}



//////////////////////////////////////////////////////////////////////
//
// CFileList
//
//////////////////////////////////////////////////////////////////////

CFileList::~CFileList()
{
   // This implementaiton of CFileList deletes the elements
   // on destruction
   while( !IsEmpty() ) 
      RemoveHead()->Release();
}



//////////////////////////////////////////////////////////////////////
//
// Helper function
//
//////////////////////////////////////////////////////////////////////

UINT __stdcall HashKey(CString key)
{
   UINT nHash = 0;
   for( int i = 0; i < key.GetLength(); i++ )
      nHash = (nHash << 5) + nHash + key[i];
   return nHash;
}

