// FileView.cpp : implementation of the CLogView class
//

#include "stdafx.h"
#include "WebPageLoader.h"

#include "Doc.h"
#include "LogView.h"
#include "MainFrm.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
// CSetRedraw helper class

class CSetRedraw
{
public:
   CSetRedraw(CListCtrl& ctl)  : list(ctl)
   { 
      list.SetRedraw(FALSE);
   }

   ~CSetRedraw() 
   { 
      list.SetRedraw(TRUE);
      list.Invalidate();
   }

private:
   CListCtrl& list;
};


/////////////////////////////////////////////////////////////////////////////
// CLogView

IMPLEMENT_DYNCREATE(CLogView, CListView)

BEGIN_MESSAGE_MAP(CLogView, CListView)
   //{{AFX_MSG_MAP(CLogView)
   ON_COMMAND(ID_EDIT_REFRESH, OnEditRefresh)
   ON_WM_CONTEXTMENU()
   ON_WM_SIZE()
   ON_COMMAND(ID_EDIT_DELETE, OnDelete)
   ON_COMMAND(ID_EDIT_PRIORITY, OnGivePriority)
   ON_COMMAND(ID_EDIT_MOVEUP, OnMoveUp)
   ON_COMMAND(ID_EDIT_MOVEDOWN, OnMoveDown)
   ON_COMMAND(ID_EDIT_RANDOMIZE, OnRandomize)   
   ON_COMMAND(ID_EDIT_COPY, OnCopy)   
   ON_COMMAND(ID_EDIT_OPENBROWSE, OnOpenBrowser)   
   ON_COMMAND(ID_EDIT_GOTOACTIVE, OnGotoActive)   
   ON_COMMAND(ID_EDIT_IGNORE_FILENAME, OnIgnoreFilename)   
   ON_COMMAND(ID_EDIT_IGNORE_DOMAIN, OnIgnoreDomain)   
   ON_UPDATE_COMMAND_UI(ID_EDIT_DELETE, OnUpdateDelete)
   ON_UPDATE_COMMAND_UI(ID_EDIT_PRIORITY, OnUpdateGivePriority)
   ON_UPDATE_COMMAND_UI(ID_EDIT_MOVEUP, OnUpdateMoveUp)
   ON_UPDATE_COMMAND_UI(ID_EDIT_MOVEDOWN, OnUpdateMoveDown)
   ON_UPDATE_COMMAND_UI(ID_EDIT_COPY, OnUpdateCopy)
   ON_UPDATE_COMMAND_UI(ID_EDIT_OPENBROWSE, OnUpdateOpenBrowser)
   ON_UPDATE_COMMAND_UI(ID_EDIT_RANDOMIZE, OnUpdateRandomize)
   ON_UPDATE_COMMAND_UI(ID_EDIT_GOTOACTIVE, OnUpdateGotoActive)
   ON_UPDATE_COMMAND_UI(ID_EDIT_IGNORE_FILENAME, OnUpdateIgnoreFilename)
   ON_UPDATE_COMMAND_UI(ID_EDIT_IGNORE_DOMAIN, OnUpdateIgnoreDomain)
   ON_NOTIFY_REFLECT(LVN_GETDISPINFO, OnGetdispInfo)
   ON_NOTIFY_REFLECT(LVN_ODCACHEHINT, OnOdcacheHint)
   //}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CLogView construction/destruction

CLogView::CLogView()
{
   m_dwStyle = ID_VIEW_LOG;
   m_iLastActive = -1;

   ::srand(::time(NULL));
}

CLogView::~CLogView()
{
}

BOOL CLogView::PreCreateWindow(CREATESTRUCT& cs)
{
   cs.style |= LVS_OWNERDATA;

   return CListView::PreCreateWindow(cs);
}


/////////////////////////////////////////////////////////////////////////////
// CLogView drawing

void CLogView::OnDraw(CDC* pDC)
{
}

void CLogView::OnInitialUpdate()
{
   CListCtrl& ctlList = GetListCtrl();

   // Create the image list for the tree control
   m_ImageList.Create(IDR_IMAGES, 16, 1, RGB(0,0,255));
   ctlList.SetImageList(&m_ImageList, LVSIL_SMALL);

   ctlList.ModifyStyle(LVS_TYPEMASK, LVS_REPORT|LVS_NOCOLUMNHEADER);

   // Add columns...
   CString s;
   s.LoadString(IDS_COL_ITEM);
   ctlList.InsertColumn(0, s, LVCFMT_LEFT);

   // Set reasonable widths for our columns
   ctlList.SetColumnWidth(0, 1000 /*LVSCW_AUTOSIZE_USEHEADER*/);

   CListView::OnInitialUpdate();
}


/////////////////////////////////////////////////////////////////////////////
// CLogView diagnostics

#ifdef _DEBUG
void CLogView::AssertValid() const
{
   CListView::AssertValid();
}

void CLogView::Dump(CDumpContext& dc) const
{
   CListView::Dump(dc);
}

CDoc* CLogView::GetDocument() // non-debug version is inline
{
   ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CDoc)));
   return (CDoc*)m_pDocument;
}
#endif //_DEBUG


/////////////////////////////////////////////////////////////////////////////
// CLogView helpers

void CLogView::_GetSelectedFiles(CSession* pSession, CArray<CDownloadFile*, CDownloadFile*&>& aFiles)
{
   CListCtrl& ctlList = GetListCtrl();
   POSITION pos = pSession->m_Files.GetHeadPosition();
   int idx = 0;
   while( pos != NULL ) {
      CDownloadFile *pFile = pSession->m_Files.GetNext(pos);
      ASSERT_VALID(pFile);
      LV_ITEM itm = { 0 };
      itm.iItem = idx;
      itm.mask = LVIF_STATE;
      itm.stateMask = LVIS_SELECTED;
      ctlList.GetItem(&itm);
      if( itm.state & LVIS_SELECTED )
         aFiles.Add(pFile);
      idx++;
   }
}

int CLogView::_FileStateToImage(const CDownloadFile* pFile)
{
   switch( pFile->m_State ) {
   case FILESTATE_CONNECTING:
      return 17;
   case FILESTATE_DOWNLOADING:
      return 3;
   case FILESTATE_PARSING:
      return 22;
   case FILESTATE_SKIPPED:
   case FILESTATE_BROKEN:
      return 15;
   case FILESTATE_ERROR:
      return 16;
   case FILESTATE_DONE:
      return 4;
   case FILESTATE_WAITING:
      return 1;
   case FILESTATE_ALREADYTHERE:
      return 21;
   }
   return 1;
}


/////////////////////////////////////////////////////////////////////////////
// CLogView list updaters

void CLogView::ClearItems()
{
   CListCtrl& ctlList = GetListCtrl();   

   CSetRedraw redraw(ctlList);

   ctlList.DeleteAllItems();
}

void CLogView::CreateItems()
{
   CListCtrl& ctlList = GetListCtrl();   
  
   CSetRedraw redraw(ctlList);

   int iSelIdx = ctlList.GetSelectionMark();

   ctlList.DeleteAllItems();

   RefreshItems();

   if( iSelIdx >= 0 ) {
      ctlList.SetItemState(iSelIdx, LVIS_SELECTED, LVIS_SELECTED);
      ctlList.EnsureVisible(iSelIdx, FALSE);
   }
}

void CLogView::CreateItemsRemaining()
{
   CListCtrl& ctlList = GetListCtrl();   
  
   CSetRedraw redraw(ctlList);

   RefreshItems();
}

void CLogView::RefreshItems()
{
   switch( m_dwStyle ) {
   case ID_VIEW_LOG:
   default:
      _RefreshLogs();
      break;
   case ID_VIEW_STATUS:
      _RefreshStatus();
      break;
   case ID_VIEW_FILES:
      _RefreshFiles();
      break;
   };
}

void CLogView::_RefreshLogs()
{
   CSession *pSession = GetDocument()->GetSelectedSession();
   if( pSession == NULL ) return;
   ASSERT_VALID(pSession);

   CSessionLogLock lock(pSession);

   CListCtrl& ctlList = GetListCtrl();

   int nCount = pSession->m_Logs.GetSize();

   if( nCount == ctlList.GetItemCount() )
      return;

   ctlList.SetItemCountEx(nCount, LVSICF_NOSCROLL);
}

void CLogView::_GetLogCache(int iFromIdx, int iToIdx)
// Get the log item details
{
   CSession *pSession = GetDocument()->GetSelectedSession();
   if( pSession == NULL ) return;
   ASSERT_VALID(pSession);

   CSessionLogLock lock(pSession);

   CListCtrl& ctlList = GetListCtrl();

   int nCount = ctlList.GetItemCount();

   for( int iIndex = iFromIdx; iIndex <= iToIdx; iIndex++ ) {
      const CLog& log = pSession->m_Logs[ nCount - iIndex - 1 ];
      CString sPage = log.sPage;
      if( log.uMsgID == IDS_LOG_FILEDOWNLOADED && sPage.Find(_T('/'), 8) > 0 )
         sPage = sPage.Mid(sPage.Find(_T('/'), 8));
      CString sLogMsg;
      sLogMsg.Format(log.uMsgID, sPage);
      CACHEITEM Item;
      Item.iItem = iIndex;
      Item.sTitle.Format(_T("%s - %s"), log.tTime.Format(_T("%X")), sLogMsg);
      Item.iImage = 12;
      m_aCache.Add(Item);
   }
}

void CLogView::_RefreshFiles()
{
   CSession *pSession = GetDocument()->GetSelectedSession();
   if( pSession == NULL ) return;
   ASSERT_VALID(pSession);

   CSessionFilesLock lock(pSession);

   BOOL bHasPriority = pSession->m_Settings.m_bUsePriorityFileNameFilter;

   CListCtrl& ctlList = GetListCtrl();

   int nCount = pSession->m_Files.GetCount();

   if( nCount != ctlList.GetItemCount() )
      ctlList.SetItemCountEx(nCount, LVSICF_NOINVALIDATEALL | LVSICF_NOSCROLL);

   int iIndex = 0;

   POSITION pos = pSession->m_Files.GetHeadPosition();
   while( pos != NULL ) {
      CDownloadFile *pFile = pSession->m_Files.GetNext(pos);
      ASSERT_VALID(pFile);

      int iImage = _FileStateToImage(pFile);
      if( pFile->m_iLastImage != iImage ) {
         pFile->m_iLastImage = iImage;
         ctlList.RedrawItems(iIndex, iIndex);
      }

      // Remember active entry.
      // Bonus feature: Don't jump to bottom of list if
      // we opted for priority and filling single pages.
      if( pFile->IsBusy() && iIndex < nCount - (bHasPriority ? 10 : 1) )
         m_iLastActive = iIndex;

      iIndex++;
   }
}

void CLogView::_GetFileCache(int iFromIdx, int iToIdx)
// Get File information
{
   CSession *pSession = GetDocument()->GetSelectedSession();
   if( pSession == NULL ) return;
   ASSERT_VALID(pSession);

   CSessionFilesLock lock(pSession);

   int iCurIdx = iFromIdx; 
   
   POSITION pos = pSession->m_Files.FindIndex(iFromIdx);
   while( pos != NULL ) {
      CDownloadFile *pFile = pSession->m_Files.GetNext(pos);
      ASSERT_VALID(pFile);

      int iImage = _FileStateToImage(pFile);

      CACHEITEM Item;
      Item.iItem = iCurIdx;
      Item.sTitle = pFile->m_sURL;
      Item.iImage = iImage;
      m_aCache.Add(Item);

      if( ++iCurIdx > iToIdx )
         break;
   }
}

void CLogView::_RefreshStatus()
// The status display is supposed to display all files
// currently in download progress. Currently it just
// show the active file download as a single entry in the list.
{
   CSession *pSession = GetDocument()->GetSelectedSession();
   if( pSession == NULL ) return;
   ASSERT_VALID(pSession);

   CSessionFilesLock lock(pSession);

   CListCtrl& ctlList = GetListCtrl();

   int nCount = 0;

   POSITION pos = pSession->m_Files.GetHeadPosition();
   while( pos != NULL ) {
      const CDownloadFile *pFile = pSession->m_Files.GetNext(pos);     
      if( pFile->IsBusy() )
         nCount++;
   }

   if( nCount != ctlList.GetItemCount() )
      ctlList.SetItemCountEx(nCount, LVSICF_NOINVALIDATEALL | LVSICF_NOSCROLL);

   ctlList.Invalidate();
}

void CLogView::_GetStatusCache(int iFromIdx, int iToIdx)
// Get status item information
{
   CSession *pSession = GetDocument()->GetSelectedSession();
   if( pSession == NULL ) return;
   ASSERT_VALID(pSession);

   extern CString GetFormattedNumber(LONGLONG n);

   CSessionFilesLock lock(pSession);

   int iCurIdx = 0;

   POSITION pos = pSession->m_Files.GetHeadPosition();
   while( pos != NULL ) {
      const CDownloadFile *pFile = pSession->m_Files.GetNext(pos);
      
      if( !pFile->IsBusy() )
         continue;

      if( iCurIdx >= iFromIdx && iCurIdx <= iToIdx ) {

         int iImage = pFile->m_bIsImage ? 9 : 0;
         if( pFile->m_State == FILESTATE_CONNECTING ) 
            iImage = (pFile->m_bIsImage ? 18 : 19);
         if( pFile->m_State == FILESTATE_PARSING ) 
            iImage = 22;
      
         UINT uRes = IDS_COL_CONNECTING;
         if( pFile->m_State == FILESTATE_DOWNLOADING ) 
            uRes = IDS_COL_STATUS;
         if( pFile->m_State == FILESTATE_PARSING ) 
            uRes = IDS_COL_PARSING;
         CString s;
         s.Format(uRes, 
            pFile->m_sURL, 
            GetFormattedNumber(pFile->m_llBytesDownloaded));

         CACHEITEM Item;
         Item.iItem = iCurIdx;
         Item.sTitle = s;
         Item.iImage = iImage;
         m_aCache.Add(Item);
      }

      if( ++iCurIdx > iToIdx )
         break;
   }
}


/////////////////////////////////////////////////////////////////////////////
// CLogView message handlers

void CLogView::OnEditRefresh() 
{
   CreateItems();
}

void CLogView::OnSize(UINT nType, int cx, int cy)
{ 
   CListCtrl& ctlList = GetListCtrl();
   ctlList.SetColumnWidth(0, cx - 30);
}

void CLogView::OnContextMenu(CWnd* pWnd, CPoint point)
{ 
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
   case ID_VIEW_STATUS:
      break;
   case ID_VIEW_LOG:
      return;
   }

   CMainFrame *frm = (CMainFrame *)::AfxGetMainWnd();
   ASSERT_VALID(frm);
   if( frm == NULL ) return;

   CMenu menu;
   menu.LoadMenu(IDR_POPUP);
   if( menu == NULL ) return;
   CMenu *editmenu = menu.GetSubMenu(0);
   if( editmenu == NULL ) return;
   editmenu->TrackPopupMenu(TPM_LEFTALIGN, 
      point.x, point.y,
      frm,
      NULL);
}

void CLogView::OnDelete()
{

   if( IDYES != AfxMessageBox(IDS_DELETE_FILES, MB_YESNO | MB_ICONQUESTION) )
      return;

   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;
   
   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   CWaitCursor wait;

   CSessionDataLock  lock1(pSession);
   CSessionFilesLock lock2(pSession);

   CListCtrl& ctlList = GetListCtrl();

   switch( m_dwStyle ) {
   case ID_VIEW_STATUS:
   case ID_VIEW_FILES:
      {
         CArray< CDownloadFile*, CDownloadFile*& > aFiles;
         _GetSelectedFiles(pSession, aFiles);
         for( int i = 0; i < aFiles.GetSize(); i++ ) {
            CDownloadFile* pFile = aFiles[i];

            CAutoRefCount ref(pFile);

            BOOL bWasOnline = FALSE;
            pFile->Abort(bWasOnline);

            POSITION pos = pSession->m_Files.Find( pFile );
            pSession->m_Files.RemoveAt( pos );

            pFile->Release();
         }
      }
      break;
   }

   ::AfxGetMainWnd()->PostMessage(WM_SCHEDULE);

   OnEditRefresh();
}

void CLogView::OnIgnoreFilename()
{
   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;

   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   CWaitCursor wait;
   
   CSessionDataLock  lock1(pSession);
   CSessionFilesLock lock2(pSession);
   
   CListCtrl& ctlList = GetListCtrl();

   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      {
         CArray< CDownloadFile*, CDownloadFile*& > aFiles;
         _GetSelectedFiles(pSession, aFiles);
         for( int i = 0; i < aFiles.GetSize(); i++ ) {
            const CDownloadFile* pFile = aFiles[i];
            CString sServer, sPage;
            DWORD dwType;
            INTERNET_PORT nPort;
            if( AfxParseURL( pFile->m_sURL, dwType, sServer, sPage, nPort ) )
               pSession->m_pPreferences->m_aBannerProviders.Add(sPage);
         }
      }
      break;
   }

   OnEditRefresh();
}

void CLogView::OnIgnoreDomain()
{
   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;

   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   CWaitCursor wait;
   
   CSessionDataLock  lock1(pSession);
   CSessionFilesLock lock2(pSession);
   
   CListCtrl& ctlList = GetListCtrl();
   
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      {
         CArray< CDownloadFile*, CDownloadFile*& > aFiles;
         _GetSelectedFiles(pSession, aFiles);
         for( int i = 0; i < aFiles.GetSize(); i++ ) {
            const CDownloadFile* pFile = aFiles[i];
            CString sServer, sPage;
            DWORD dwType;
            INTERNET_PORT nPort;
            if( AfxParseURL( pFile->m_sURL, dwType, sServer, sPage, nPort ) )
               pSession->m_pPreferences->m_aBannerProviders.Add(sServer);
         }
      }
      break;
   }

   OnEditRefresh();
}

void CLogView::OnGivePriority()
{
   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;

   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   CWaitCursor wait;

   CSessionFilesLock lock(pSession);

   // To be able to set priority, we need to
   // enable the feature in the Session settings.
   pSession->m_Settings.m_bUseImmediateDownload = TRUE;

   CListCtrl& ctlList = GetListCtrl();
   
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      {
         CArray< CDownloadFile*, CDownloadFile*& > aFiles;
         _GetSelectedFiles(pSession, aFiles);
         for( int i = 0; i < aFiles.GetSize(); i++ ) {
            CDownloadFile* pFile = aFiles[i];
            pFile->m_iPriority = 20;
         }
      }
      break;
   }

   OnEditRefresh();
}

void CLogView::OnMoveUp()
{
   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;

   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   CWaitCursor wait;

   CSessionFilesLock lock(pSession);
   
   CListCtrl& ctlList = GetListCtrl();

   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      {
         CArray< CDownloadFile*, CDownloadFile*& > aFiles;
         _GetSelectedFiles(pSession, aFiles);
         for( int i = 0; i < aFiles.GetSize(); i++ ) {
            CDownloadFile* pFile = aFiles[i];
            if( pFile->IsBusy() )
               continue;
            POSITION pos = pSession->m_Files.Find( pFile );
            pSession->m_Files.RemoveAt( pos );
            pSession->m_Files.AddHead( pFile );
         }
      }
      break;
   }

   OnEditRefresh();
}

void CLogView::OnMoveDown()
{
   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;

   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   CWaitCursor wait;

   CSessionFilesLock lock(pSession);
   
   CListCtrl& ctlList = GetListCtrl();
   
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      {
         CArray< CDownloadFile*, CDownloadFile*& > aFiles;
         _GetSelectedFiles(pSession, aFiles);
         for( int i = 0; i < aFiles.GetSize(); i++ ) {
            CDownloadFile* pFile = aFiles[i];
            if( pFile->IsBusy() )
               continue;
            POSITION pos = pSession->m_Files.Find( pFile );
            pSession->m_Files.RemoveAt( pos );
            pSession->m_Files.AddTail( pFile );
         }
      }
      break;
   }

   OnEditRefresh();
}

void CLogView::OnRandomize()
{
   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;

   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   CWaitCursor wait;

   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pSession->RandomizeFiles(2000);
      break;
   }

   OnEditRefresh();
}

void CLogView::OnCopy()
{
   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;

   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   CWaitCursor wait;

   CListCtrl& ctlList = GetListCtrl();

   POSITION pos = ctlList.GetFirstSelectedItemPosition();
   if( pos == NULL ) return;

   CString sText = ctlList.GetItemText(ctlList.GetNextSelectedItem(pos), 0);

   if( !OpenClipboard() ) 
      return;
   EmptyClipboard();
   HGLOBAL hGlobal = ::GlobalAlloc(GMEM_MOVEABLE, sText.GetLength() + 1);
   if( hGlobal != NULL )  {
#ifndef _UNICODE      
      strcpy( (char *)GlobalLock(hGlobal), static_cast<LPCSTR>(sText));
      GlobalUnlock(hGlobal);
      SetClipboardData(CF_TEXT, hGlobal);
#else
      wcscpy( (wchar_t *)GlobalLock(hGlobal), static_cast<LPCWSTR>(sText));
      GlobalUnlock(hGlobal);
      SetClipboardData(CF_UNICODETEXT, hGlobal);
#endif // hGlobal
   }
   CloseClipboard();
}

void CLogView::OnOpenBrowser()
{
   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;

   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   CWaitCursor wait;
   
   CListCtrl& ctlList = GetListCtrl();

   POSITION pos = ctlList.GetFirstSelectedItemPosition();
   if( pos == NULL ) return;

   CString sText = ctlList.GetItemText(ctlList.GetNextSelectedItem(pos), 0);
   ::ShellExecute(m_hWnd, _T("open"), sText, NULL, NULL, SW_SHOW);
}

void CLogView::OnGotoActive()
{
   CDoc *pDoc = GetDocument();
   ASSERT_KINDOF(CDoc,pDoc);
   if( pDoc == NULL ) return;
   
   CSession *pSession = pDoc->GetSelectedSession();
   ASSERT_VALID(pSession);
   if( pSession == NULL ) return;

   if( m_dwStyle != ID_VIEW_FILES ) 
      return;

   CWaitCursor wait;

   if( m_iLastActive < 0 ) {
      ::MessageBeep((UINT)-1);
      return;
   }

   CListCtrl& ctlList = GetListCtrl();

   ctlList.SetItemState(-1, 0, LVNI_SELECTED);
   ctlList.SetItemState(m_iLastActive, LVNI_SELECTED, LVNI_SELECTED);
   ctlList.EnsureVisible(m_iLastActive, FALSE);
}

void CLogView::OnUpdateDelete(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
   case ID_VIEW_STATUS:
      pCmdUI->Enable( ctlList.GetSelectedCount() > 0 );
      break;
   case ID_VIEW_LOG:
      pCmdUI->Enable(FALSE);
      return;
   }
}

void CLogView::OnUpdateIgnoreFilename(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pCmdUI->Enable( ctlList.GetSelectedCount() > 0 );
      break;
   case ID_VIEW_LOG:
   case ID_VIEW_STATUS:
      pCmdUI->Enable(FALSE);
      return;
   }
}

void CLogView::OnUpdateIgnoreDomain(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pCmdUI->Enable( ctlList.GetSelectedCount() > 0 );
      break;
   case ID_VIEW_LOG:
   case ID_VIEW_STATUS:
      pCmdUI->Enable(FALSE);
      return;
   }
}

void CLogView::OnUpdateGivePriority(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pCmdUI->Enable( ctlList.GetSelectedCount() > 0 );
      break;
   case ID_VIEW_LOG:
   case ID_VIEW_STATUS:
      pCmdUI->Enable(FALSE);
      return;
   }
}

void CLogView::OnUpdateMoveUp(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pCmdUI->Enable( ctlList.GetSelectedCount() > 0 );
      break;
   case ID_VIEW_LOG:
   case ID_VIEW_STATUS:
      pCmdUI->Enable(FALSE);
      return;
   };
}

void CLogView::OnUpdateMoveDown(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pCmdUI->Enable( ctlList.GetSelectedCount() > 0 );
      break;
   case ID_VIEW_LOG:
   case ID_VIEW_STATUS:
      pCmdUI->Enable(FALSE);
      return;
   };
}

void CLogView::OnUpdateRandomize(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pCmdUI->Enable( ctlList.GetSelectedCount() > 0 );
      break;
   case ID_VIEW_LOG:
   case ID_VIEW_STATUS:
      pCmdUI->Enable(FALSE);
      return;
   }
}

void CLogView::OnUpdateCopy(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pCmdUI->Enable( ctlList.GetSelectedCount() > 0 );
      break;
   case ID_VIEW_LOG:
   case ID_VIEW_STATUS:
      pCmdUI->Enable(FALSE);
      return;
   }
}

void CLogView::OnUpdateOpenBrowser(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pCmdUI->Enable( ctlList.GetSelectedCount() > 0 );
      break;
   case ID_VIEW_LOG:
   case ID_VIEW_STATUS:
      pCmdUI->Enable(FALSE);
      return;
   }
}

void CLogView::OnUpdateGotoActive(CCmdUI* pCmdUI) 
{
   CListCtrl& ctlList = GetListCtrl();
   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      pCmdUI->Enable( TRUE );
      break;
   case ID_VIEW_LOG:
   case ID_VIEW_STATUS:
      pCmdUI->Enable(FALSE);
      return;
   }
}

void CLogView::OnOdcacheHint(NMHDR* pNMHDR, LRESULT* pResult) 
{
   NMLVCACHEHINT* pCacheHint = (NMLVCACHEHINT*)pNMHDR;

   m_aCache.RemoveAll();

   switch( m_dwStyle ) {
   case ID_VIEW_FILES:
      _GetFileCache(pCacheHint->iFrom, pCacheHint->iTo);
      break;
   case ID_VIEW_STATUS:
      _GetStatusCache(pCacheHint->iFrom, pCacheHint->iTo);
      break;
   case ID_VIEW_LOG:
      _GetLogCache(pCacheHint->iFrom, pCacheHint->iTo);
      return;
   }

   *pResult = 0;
}

void CLogView::OnGetdispInfo(NMHDR* pNMHDR, LRESULT* pResult) 
{
   LV_DISPINFO* pDispInfo = (LV_DISPINFO*) pNMHDR;

   LV_ITEM* pItem = &(pDispInfo)->item;

   int i = 0;

   // Is not in cache? Then add it now...
   BOOL bFound = FALSE;
   for( i = 0; i < m_aCache.GetSize(); i++ ) {
      if( m_aCache[i].iItem != pItem->iItem )
         continue;
      bFound = TRUE;
      break;
   }
   if( !bFound ) {
      switch( m_dwStyle ) {
      case ID_VIEW_FILES:
         _GetFileCache(pItem->iItem, pItem->iItem);
         break;
      case ID_VIEW_STATUS:
         _GetStatusCache(pItem->iItem, pItem->iItem);
         break;
      case ID_VIEW_LOG:
         _GetLogCache(pItem->iItem, pItem->iItem);
         return;
      }
   }

   // Return info...
   for( i = 0; i < m_aCache.GetSize(); i++ ) {
      if( m_aCache[i].iItem != pItem->iItem )
         continue;

      if( (pItem->mask & LVIF_TEXT) != 0 ) {
         ::ZeroMemory(pItem->pszText, pItem->cchTextMax);
         _tcsncpy(pItem->pszText, m_aCache[i].sTitle, pItem->cchTextMax - 1);
      }
      
      if( (pItem->mask & LVIF_IMAGE) != 0 )
         pItem->iImage = m_aCache[i].iImage;

      if( m_dwStyle == ID_VIEW_FILES )
         pItem->mask |= LVIF_DI_SETITEM;

      break;
   }

   *pResult = 0;
}

