
#include "stdafx.h"

#include "SiteMapCtrl.h"
#include "SiteMap.h"
#include "Items.h"


// Draw connecting lines.
// Also responsible for updating the HasChildren flag.
void CSiteMap::_DrawConnector(CDCHandle &dc, tNode *pItems, long lSize, long Index) const
{
   tNode &item = pItems[Index];
   for( int i=0; i<lSize; i++ ) {         
      if( pItems[i].ParentID==item.ID ) {        
         item.bHasChildren = true;
         if( item.bShowChildren ) {
            tNode &target = pItems[i];
            POINT ptTop = { (item.width/2) + item.x, item.y+item.height };
            POINT ptBottom = { (target.width/2) + target.x, target.y };
            POINT ptMiddle = { (ptTop.x/2) + (ptBottom.x/2), (ptTop.y/2) + (ptBottom.y/2) };
            POINT p[4] = 
            { 
               { ptTop.x, ptTop.y },
               { ptTop.x, ptMiddle.y },
               { ptBottom.x, ptMiddle.y },
               { ptBottom.x, ptBottom.y }
            };
            dc.Polyline(p,4);

            _DrawConnector(dc, pItems, lSize, i);
         }
      }
   }
};

// Draws a box with text and open/close symbol if needed...
void CSiteMap::_DrawItem(CDCHandle &dc, tNode *pItems, long lSize, long index) const
{
   ATLASSERT(pItems);
   tNode &item = pItems[index];

   // Paint box and frame
   RECT rc = {item.x, item.y, item.x+item.width, item.y+item.height};
   dc.FillRect(&rc, (HBRUSH)::GetStockObject(WHITE_BRUSH));
   dc.SelectPen(item.bIsSelected ? m_penSelected : m_penBorder);
   dc.Rectangle(rc.left, rc.top, rc.right, rc.bottom);

   // Paint shadow
   dc.SelectPen(m_penShadow);
   dc.Rectangle(item.x+item. width+1, item.y+4,    item.x+item.width+2, item.y+item.height+2);
   dc.Rectangle(item.x+4,    item.y+item.height+1, item.x+item.width+1, item.y+item.height+2);

   // Display expand/collapse symbol
   if( (m_vbAllowCollapse==VARIANT_TRUE) && item.bHasChildren ) {
      long lMiddle = (rc.left/2) + (rc.right/2);
      RECT rcKey = 
      { 
         lMiddle-(EXPANDER_SIZE/2), 
         rc.bottom-(EXPANDER_SIZE/2), 
         lMiddle+(EXPANDER_SIZE/2), 
         rc.bottom+(EXPANDER_SIZE/2) 
      };
      dc.FillRect(&rcKey, m_brushBorder);
      dc.SelectStockPen(WHITE_PEN);
      if( !item.bShowChildren ) {
         dc.MoveTo(lMiddle, rc.bottom-2);
         dc.LineTo(lMiddle, rc.bottom+3);
      }
      dc.MoveTo(lMiddle-2, rc.bottom);
      dc.LineTo(lMiddle+3, rc.bottom);
   }

   // Paint text area
   RECT rcText = 
   { 
      item.x+TEXTINSET_SIZE, 
      item.y+TEXTINSET_SIZE, 
      item.x+item.width-TEXTINSET_SIZE, 
      item.y+TEXTINSET_SIZE+TEXTAREA_HEIGHT 
   };
   dc.FillRect(&rcText, m_brushFill);

   // Draw text
   dc.SelectPen(m_penFore);
   USES_CONVERSION;
   LPCTSTR pszText = OLE2CT(item.Title);
   if( pszText!=NULL && _tcslen(pszText)!=0 ) {
      dc.ExtTextOut(
         (rcText.left + rcText.right) / 2, 
         (rcText.top + rcText.bottom) / 2, 
         ETO_CLIPPED,
         &rcText,
         pszText, 
         lstrlen(pszText),
         NULL);
   }

   if( item.bShowChildren && item.bHasChildren ) {
      for( int i=0; i<lSize; i++ ) {
         if( pItems[i].ParentID==item.ID ) _DrawItem(dc, pItems, lSize, i);
      }
   }
}

HRESULT CSiteMap::OnDraw(ATL_DRAWINFO &di)
{
   CDCHandle dc(di.hdcDraw);

   // If someone requests a re-arrangement of the graphs, this
   // is the time to do it...
   // Placing it this late ensure that several property changes on IItem,
   // which all request new layout, doesn't cause the layout algorithm to be
   // called repeatedly!
   if( m_bNeedsArragement ) {
      _Arrange();
      _CalcClientArea();
   }

   if( m_bRecalcGdi ) {
      COLORREF clr;
      //
      ::OleTranslateColor(m_clrForeColor, NULL, &clr);
      if( !m_penFore.IsNull() ) m_penFore.DeleteObject();
      m_penFore.CreatePen(PS_SOLID,0,clr);
      //
      ::OleTranslateColor(m_clrBorderColor, NULL, &clr);
      if( !m_penBorder.IsNull() ) m_penBorder.DeleteObject();
      m_penBorder.CreatePen(PS_SOLID,0,clr);
      if( !m_brushBorder.IsNull() ) m_brushBorder.DeleteObject();
      m_brushBorder.CreateSolidBrush(clr);
      //
      ::OleTranslateColor(m_clrSelectedColor, NULL, &clr);
      if( !m_penSelected.IsNull() ) m_penSelected.DeleteObject();
      m_penSelected.CreatePen(PS_SOLID,0,clr);
      //
      ::OleTranslateColor(m_clrShadowColor, NULL, &clr);
      if( !m_penShadow.IsNull() ) m_penShadow.DeleteObject();
      m_penShadow.CreatePen(PS_SOLID,0,clr);
      //
      ::OleTranslateColor(m_clrBackColor, NULL, &clr);
      if( !m_brushBack.IsNull() ) m_brushBack.DeleteObject();
      m_brushBack.CreateSolidBrush(clr);
      //
      ::OleTranslateColor(m_clrFillColor, NULL, &clr);
      if( !m_brushFill.IsNull() ) m_brushFill.DeleteObject();
      m_brushFill.CreateSolidBrush(clr);
      //
      CComQIPtr<IFont,&IID_IFont> pFont(m_pFont);
      if( pFont!=NULL ) {
         HFONT hFont;
         pFont->get_hFont(&hFont);
         m_fontText = hFont;
      };
      //
      m_bRecalcGdi = false;
   }

   // Clear window
   RECT& rc = *(RECT*)di.prcBounds;
   dc.FillRect(&rc, m_brushBack);

   // Collect simple C++ array of items
   long lSize = m_pItems->m_coll.size();
   if( lSize==0 ) return S_OK;
   tNode *pItems = new tNode[lSize];
   ATLASSERT(pItems);
   if( pItems==NULL ) return E_OUTOFMEMORY;

   _ConvertToTree(m_pItems, pItems, lSize, TRUE);


   // Translate coords
   dc.SetWindowOrg(m_ptViewPos.x, m_ptViewPos.y, NULL);

   // Paint stuff 
   HPEN penOld = dc.SelectStockPen(BLACK_PEN);
   HFONT fontOld = dc.SelectFont(m_fontText);
   COLORREF clrOldBk = dc.SetBkColor(m_clrFillColor);
   UINT taOld = dc.SetTextAlign(TA_CENTER|TA_BASELINE);

   // For all roots, draw sub-tree...
   for( int i=0; i<lSize; i++ ) {
      if( pItems[i].ParentID==-1 ) {
         _DrawConnector(dc, pItems, lSize, i);
         _DrawItem(dc, pItems, lSize, i);
      }
   }

   if( !di.bOptimize ) {
      dc.SetTextAlign(taOld);
      dc.SelectPen(penOld);
      dc.SelectFont(fontOld);
      dc.SetBkColor(clrOldBk);
   }

   delete [] pItems;

   return S_OK;
}
