
#include "stdafx.h"

#include "SiteMap.h"
#include "Layer.h"
#include "Items.h"
#include "Item.h"


/////////////////////////////////////////////////////////////////////////////
// CDragDropLayer Drag and drop

void CDragDropLayer::BeginDrag(CComPtr<IItem> &spItem, LPARAM lParam)
{
   m_spDraggedItem = spItem;
   m_ptOrgMousePos = MAKEPOINTS(lParam);
   m_lDragItemIndex = -1;
   SetCapture();
   ::SetCursor(::LoadCursor(_Module.GetResourceInstance(), (LPCTSTR)IDC_DRAG));
   m_bIsDragging = true;
}

void CDragDropLayer::EndDrag(LPARAM /*lParam*/)
{
   ATLASSERT(m_spDraggedItem);
   // Attach dragged item to new parent (remove is nessecary)
   CComPtr<IItem> spItem;
   if( m_lAttachItemIndex>=0 ) {
     m_pSite->m_pItems->get_Item(m_lAttachItemIndex + 1, &spItem);
   }
   // Set the new position...
   // The order of calls here is important because it is very difficult
   // to set a new absoulte position.
   // The main problem is that the nodes gets moved to the right if they have
   // visible children. To move a node, it cannot have a parent.
   long x,y;
   m_spDraggedItem->get_Left(&x);
   m_spDraggedItem->get_Top(&y);
   m_spDraggedItem->put_Parent(spItem);
   m_spDraggedItem->put_Left(x + m_ptOldTreeOffset.x);
   m_spDraggedItem->put_Top(y + m_ptOldTreeOffset.y);
   if( spItem!=NULL ) spItem->put_Expanded(VARIANT_TRUE); // show new child
   m_spDraggedItem->put_Selected(VARIANT_TRUE);
   //
   m_pSite->Arrange();
   CancelDrag();
}

void CDragDropLayer::CancelDrag()
{
   ReleaseCapture();
   m_spDraggedItem = NULL;
   m_bIsDragging = false;
}


/////////////////////////////////////////////////////////////////////////////
// Tree drawing

// Draws a box with text and open/close symbol if needed...
// Updates the "bHasChildren" member.
void CDragDropLayer::DrawConnector(CDCHandle &dc, tNode *pItems, long lSize, 
                                  long Index, 
                                  long xoffset, long yoffset) const
{
   tNode &item = pItems[Index];
   for( int i=0; i<lSize; i++ ) {         
      if( pItems[i].ParentID==item.ID ) {        
         item.bHasChildren = true;
         if( item.bShowChildren ) {
            tNode &target = pItems[i];
            POINT ptTop = { (item.width/2) + item.x, item.y+item.height };
            POINT ptBottom = { (target.width/2) + target.x, target.y };
            POINT ptMiddle = { (ptTop.x/2) + (ptBottom.x/2), (ptTop.y/2) + (ptBottom.y/2) };
            POINT p[4] = 
            { 
               { ptTop.x+xoffset, ptTop.y+yoffset },
               { ptTop.x+xoffset, ptMiddle.y+yoffset },
               { ptBottom.x+xoffset, ptMiddle.y+yoffset },
               { ptBottom.x+xoffset, ptBottom.y+yoffset }
            };
            dc.Polyline(p,4);

            DrawConnector(dc, pItems, lSize, i, xoffset, yoffset);
         }
      }
   }
}

// Draws the item box...
void CDragDropLayer::DrawItem(CDCHandle &dc, tNode *pItems, long lSize, 
                             long Index, 
                             long xoffset, long yoffset) const
{
   tNode &item = pItems[Index];

   // Paint box and frame
   RECT rc = {item.x, item.y, item.x+item.width, item.y+item.height};
   ::OffsetRect(&rc, xoffset, yoffset);
   POINT p[5] = 
   { 
      { rc.left, rc.top },
      { rc.right, rc.top },
      { rc.right, rc.bottom },
      { rc.left, rc.bottom },
      { rc.left, rc.top }
   };
   dc.Polyline(p,5);

   if( item.bShowChildren && item.bHasChildren ) {
      for( int i=0; i<lSize; i++ ) {
         if( pItems[i].ParentID==item.ID ) DrawItem(dc, pItems, lSize, i, xoffset, yoffset);
      }
   }
}

void CDragDropLayer::DrawNewConnector(CDCHandle &dc, tNode *pNodes, 
                                     long ParentIndex,
                                     long ChildIndex,
                                     POINT &ptOffset) const
{
   // Construct our own little Node array, so we can draw the connection
   // between dragged item and new parent...
   tNode pConnectedNodes[2];
   // Make a copy of the two nodes we wish to connect
   pConnectedNodes[0] = pNodes[ParentIndex];
   pConnectedNodes[1] = pNodes[ChildIndex];
   // Manipulate their properties so they match the new Node array
   // and make sure they display...
   pConnectedNodes[0].ID = 0;
   pConnectedNodes[0].ParentID = -1;
   pConnectedNodes[0].x -= ptOffset.x + m_pSite->m_ptViewPos.x;
   pConnectedNodes[0].y -= ptOffset.y + m_pSite->m_ptViewPos.y;
   pConnectedNodes[0].bShowChildren = true;
   pConnectedNodes[1].ID = 1;
   pConnectedNodes[1].ParentID = 0;
   DrawConnector(dc, pConnectedNodes, 2, 0, ptOffset.x, ptOffset.y);
}

HRESULT CDragDropLayer::DrawTree(HDC hdc, POINT ptOffset)
{
   ATLASSERT(m_pSite);
   ATLASSERT(m_spDraggedItem);

   CDCHandle dc(hdc);

   // Back-up settings, so we can redraw (erase) the drawn tree...
   m_ptOldTreeOffset = ptOffset;

   // Translate coords
   ptOffset.x -= m_pSite->m_ptViewPos.x;
   ptOffset.y -= m_pSite->m_ptViewPos.y;

   // Collect the C++ array of items
   long lSize = m_pSite->m_pItems->m_coll.size();
   tNode *pNodes = new tNode[lSize];
   if( pNodes==NULL ) return E_OUTOFMEMORY;

   m_pSite->_ConvertToTree(m_pSite->m_pItems, pNodes, lSize);

   // If the Dragged Node's Index hasn't been determined,
   // then do it now...
   if( m_lDragItemIndex==-1 ) {
      long CurID;
      m_spDraggedItem->get_ID(&CurID);
      for( int i=0; i<lSize; i++ ) {
         if( pNodes[i].ID==CurID ) {
            m_lDragItemIndex = i;
            break;
         }
      }
      ATLASSERT(m_lDragItemIndex>=0);
      if( m_lDragItemIndex==-1 ) return E_FAIL;
   }

   if( m_lDragItemIndex>=0 ) {
      // Paint stuff 
      CPen m_pen;
      m_pen.CreatePen(PS_DOT, 1, RGB(0,0,0));
      dc.SetROP2(R2_NOTXORPEN);
      HPEN penOld = dc.SelectPen(m_pen);
      DrawConnector(dc, pNodes, lSize, m_lDragItemIndex, ptOffset.x, ptOffset.y);
      DrawItem(dc, pNodes, lSize, m_lDragItemIndex, ptOffset.x, ptOffset.y);

      if( HitTestMap(pNodes, lSize, ptOffset, &m_lAttachItemIndex) ) {
         ATLASSERT(m_lAttachItemIndex>=0);
         DrawNewConnector(dc, pNodes, m_lAttachItemIndex, m_lDragItemIndex, ptOffset);
      }

      dc.SelectPen(penOld);
   }


   delete [] pNodes;
   return S_OK;
}


/////////////////////////////////////////////////////////////////////////////
// Drag Hit Test

void CDragDropLayer::TraverseHitMap(tNode *pNodes, long lSize, long Index, 
                                   POINT &pt, 
                                   long *pCloseness,
                                   long *pHitIndex) const
{
   enum { HIT_OUTSET_X = 30, HIT_OUTSET_Y = 50 };

   tNode &node = pNodes[Index];
   if( node.ID==m_lDragItemIndex ) return; // don't scan self or children

   RECT rc = 
   { 
      node.x - node.width - HIT_OUTSET_X, 
      node.y + node.height, 
      node.x + node.width + HIT_OUTSET_X, 
      node.y + node.height + HIT_OUTSET_Y 
   };
   if( ::PtInRect(&rc, pt) ) {
      // Point inside the hit box of the Node.
      // Now figure out if it is closer than any of the other hits.
      // Choosing the closest node works great for a good drag'n'drop feel.
      POINT ptMiddle = { rc.left + ((rc.right-rc.left)/2), rc.top + ((rc.bottom-rc.top)/2) };
      long radius = ((ptMiddle.x-pt.x)*(ptMiddle.x-pt.x)) + 
                    ((ptMiddle.y-pt.y)*(ptMiddle.y-pt.y));
      if( radius<*pCloseness ) {
         *pCloseness = radius;
         *pHitIndex = Index;
      }
   }
   // Scan all children unless they are children of the dragged item...
   if( (node.bShowChildren) && (node.ID!=m_lDragItemIndex) ) {
      for( int i=0; i<lSize; i++ ) {
         if( pNodes[i].ParentID==node.ID ) {
            TraverseHitMap(pNodes, lSize, i, pt, pCloseness, pHitIndex);
         }
      }
   }
}

// Test if point is inside any of the Hit Rectangles of
// the valid nodes.
// The Hit Rectangle is just below each Node.
// Valid nodes are all nodes except children of the
// dragged item.
bool CDragDropLayer::HitTestMap(tNode *pNodes, long lSize, 
                               POINT ptOffset, 
                               long *pHitIndex) const
{
   ATLASSERT(m_bIsDragging);
   ATLASSERT(!::IsBadReadPtr(pNodes,lSize*sizeof(tNode)));
   ATLASSERT(m_lDragItemIndex>=0);
   ATLASSERT(pHitIndex);

   // We didn't find anything yet...
   *pHitIndex = -1;

   // Translate coords
   ptOffset.x += m_pSite->m_ptViewPos.x;
   ptOffset.y += m_pSite->m_ptViewPos.y;

   // Scan all roots...
   tNode &drag = pNodes[m_lDragItemIndex];
   POINT pt = { drag.x + ptOffset.x, drag.y + ptOffset.y };
   long lCloseness = 0x7FFFFFFF;
   for( int i=0; i<lSize; i++ ) {
      if( pNodes[i].ParentID==-1 ) {
         TraverseHitMap(pNodes, lSize, i, pt, &lCloseness, pHitIndex);
      }
   }
   return *pHitIndex>=0;
}
