#if !defined(AFX_SQLPARSER_H__20010927_3552_9275_D2A6_0080AD509054__INCLUDED_)
#define AFX_SQLPARSER_H__20010927_3552_9275_D2A6_0080AD509054__INCLUDED_

#pragma once

#ifndef __ATLBASE_H__
   #error This file requires atlbase.h to be included first
#endif


/////////////////////////////////////////////////////////////////////////////
// Types and structures

#define MAX_TOKEN_STRLEN 64

enum {
   ERR_NOTSTRING,
   ERR_NOTBOOLEAN,
   ERR_UNKNOWNKEYWORD,
   ERR_UNEXPECTED,
   ERR_COLUMNEXPECTED,
   ERR_TABLEEXPECTED,
   ERR_DELIMTEREXPECTED,
   ERR_TOOLONG,
   ERR_TYPEMISMATCH,
   ERR_SYNTAX,
   ERR_ENDEXPECTED
};

typedef enum {
   OP_END = 0,
   OP_KEYWORD,
   OP_SELECT,
   OP_UPDATE,
   OP_FROM,
   OP_SET,
   OP_WHERE,
   OP_OPERATOR,
   OP_AND,
   OP_OR,
   OP_PAREN_L,
   OP_PAREN_R,
   OP_ARITH,
   OP_COMMA,
   OP_TABLE,
   OP_FIELD,
   OP_STRING,
   OP_INTEGER,
} OPTYPE;

struct TOKEN
{
   TOKEN(OPTYPE t)
   {
      type = t;
      value[0] = _T('\0');
   }
   TOKEN(OPTYPE t, LPCTSTR pstr)
   {
      type = t;
      _tcscpy( value, pstr );
   }
   TOKEN(OPTYPE t, TCHAR c)
   {
      type = t;
      value[0] = c;
      value[1] = _T('\0');
   }
   OPTYPE type;
   TCHAR value[MAX_TOKEN_STRLEN];
};

struct CMDOP
{
   CMDOP(OPTYPE t, CMDOP* pLeft=NULL, CMDOP* pRight=NULL)
   {
      type = t;
      left = pLeft;
      right = pRight;
      value[0] = _T('\0');
   }
   CMDOP(TOKEN &t, CMDOP* pLeft=NULL, CMDOP* pRight=NULL)
   {
      type = t.type;
      left = pLeft;
      right = pRight;
      _tcscpy( value, t.value );
   }
   ~CMDOP()
   {
      if( left ) delete left;
      if( right ) delete right;
   }
   OPTYPE type;
   CMDOP* left;
   CMDOP* right;
   TCHAR value[MAX_TOKEN_STRLEN];
};
typedef CMDOP* PCMDOP;


/////////////////////////////////////////////////////////////////////////////
// CEvaluator

struct EVALPROP
{
   TCHAR szName[MAX_TOKEN_STRLEN];
   TCHAR szValue[MAX_TOKEN_STRLEN];
};
typedef EVALPROP* PEVALPROP;

class CEvaluator
{
private:
   PEVALPROP m_pProps;
   int m_cProps;

public:
   BOOL Evaluate(const PCMDOP pTree, const PEVALPROP pProps, int cProps)
   {
      if( pTree==NULL ) return TRUE; // No WHERE clause - match all
      if( pTree->type != OP_WHERE ) throw ERR_UNEXPECTED; // Must be WHERE clause?!
      PCMDOP pWhereTree = pTree->left;
      if( pTree==NULL ) throw ERR_UNEXPECTED;
      // Evaluate parse tree.
      // We use VARIANTs - which are slow - but easy to manage.
      m_pProps = pProps;
      m_cProps = cProps;
      CComVariant v = Descent(pWhereTree);
      if( v.vt==VT_BOOL ) return v.boolVal==VARIANT_TRUE;
      throw ERR_SYNTAX;
   }
   CComVariant GetValue(const PCMDOP pTree, const PEVALPROP pProps, int cProps)
   {
      if( pTree==NULL ) return CComVariant();
      m_pProps = pProps;
      m_cProps = cProps;
      return Descent(pTree);
   }

private:
   CComVariant Descent(const PCMDOP pTree) const
   {
      switch( pTree->type ) {
      case OP_OPERATOR:
         {
            CComVariant left = Descent(pTree->left);
            CComVariant right = Descent(pTree->right);
            if( _tcscmp(pTree->value, _T("="))==0 ) return left==right ? true : false;
            if( _tcscmp(pTree->value, _T("<"))==0 ) return left<right ? true : false;
            if( _tcscmp(pTree->value, _T(">"))==0 ) return left>right ? true : false;
            if( _tcscmp(pTree->value, _T("<="))==0 ) return left>right ? false: true;
            if( _tcscmp(pTree->value, _T(">="))==0 ) return left<right ? false : true;
            throw ERR_TYPEMISMATCH;
         }
      case OP_ARITH:
         {
            CComVariant left = Descent(pTree->left);
            CComVariant right = Descent(pTree->right);
            CComVariant res;
            if( _tcscmp(pTree->value, _T("+"))==0 ) ::VarAdd(&left, &right, &res);
            else if( _tcscmp(pTree->value, _T("-"))==0 ) ::VarSub(&left, &right, &res);
            else if( _tcscmp(pTree->value, _T("*"))==0 ) ::VarMul(&left, &right, &res);
            else if( _tcscmp(pTree->value, _T("/"))==0 ) ::VarDiv(&left, &right, &res);
            if( left.vt!=res.vt ) throw ERR_TYPEMISMATCH;
            return res;
         }
      case OP_AND:
      case OP_OR:
         {
            CComVariant left = Descent(pTree->left);
            CComVariant right = Descent(pTree->right);
            CComVariant res;
            switch( pTree->type ) {
            case OP_AND: 
               ::VarAnd(&left, &right, &res);
               break;
            case OP_OR:
               ::VarOr(&left, &right, &res);
               break;
            }
            if( left.vt!=res.vt ) throw ERR_TYPEMISMATCH;
            return res;
         }
      case OP_INTEGER:
         return _ttol(pTree->value);
      case OP_STRING:
         return pTree->value;
      case OP_KEYWORD:
         {
            for( int i=0; i<m_cProps; i++ ) {
               if( _tcsicmp(m_pProps[i].szName, pTree->value)==0 ) return m_pProps[i].szValue;
            }
            throw ERR_UNKNOWNKEYWORD;
         }
      default:
         throw ERR_SYNTAX;
      }
   }
};


/////////////////////////////////////////////////////////////////////////////
// CLexer

class CLexer
{
private:
   LPCTSTR m_pstr;

public:
   PCMDOP Lex(LPCTSTR pstrCommand)
   {
      m_pstr = pstrCommand;
      return LexSQL();
   }

private:
   PCMDOP LexSQL()
   {
      TOKEN t = GetToken();
      switch( t.type ) {
      case OP_SELECT:
         {
            PCMDOP pFields = GetFieldsClause();
            PCMDOP pFrom = GetFromClause();
            if( GetToken().type!=OP_END ) throw ERR_ENDEXPECTED;
            return new CMDOP(t, pFields, pFrom);
         }
      case OP_UPDATE:
         {
            TOKEN table = GetToken(); // a single table select is supported
            if( table.type!=OP_KEYWORD ) throw ERR_TABLEEXPECTED;
            PCMDOP pUpdate = GetUpdateClause();
            if( GetToken().type!=OP_END ) throw ERR_ENDEXPECTED;
            return new CMDOP(t, new CMDOP(table), pUpdate);
         }
      default:
         throw ERR_SYNTAX;
      }
   }
   PCMDOP GetFieldsClause()
   {
      TOKEN t = GetToken();
      if( t.type==OP_ARITH && t.value[0]==_T('*') ) return new CMDOP(t);
      if( t.type!=OP_KEYWORD ) throw ERR_COLUMNEXPECTED;
      PCMDOP pCmd = new CMDOP(t);
      PCMDOP pTopCmd = pCmd;
      while( PeekTokenType()==OP_COMMA ) {
         GetToken(); // comma
         PCMDOP pNewCmd = new CMDOP(GetToken());
         pTopCmd->left = pNewCmd;
         pTopCmd = pNewCmd;
      }
      return pCmd;
   }
   PCMDOP GetUpdateClause()
   {
      TOKEN t = GetToken();
      if( t.type!=OP_SET ) throw ERR_SYNTAX;
      PCMDOP pAssign = GetAssignClause();
      PCMDOP pWhere = GetWhereClause();
      return new CMDOP(t, pAssign, pWhere);
   }
   PCMDOP GetAssignClause()
   {
      TOKEN t = GetToken();
      if( t.type!=OP_KEYWORD ) throw ERR_COLUMNEXPECTED;
      TOKEN a = GetToken();
      if( a.type!=OP_ARITH && a.value[0]!=_T('=') ) throw ERR_SYNTAX;
      return new CMDOP(a, new CMDOP(t), GetExpr1());
   }
   PCMDOP GetFromClause()
   {
      if( GetToken().type!=OP_FROM ) throw ERR_SYNTAX;
      TOKEN t = GetToken(); // a single table select is supported
      if( t.type!=OP_KEYWORD ) throw ERR_TABLEEXPECTED;
      return new CMDOP(OP_FROM, new CMDOP(t), GetWhereClause());
   }
   PCMDOP GetWhereClause()
   {
      if( GetToken().type!=OP_WHERE ) return NULL; // The WHERE clause need not be there!
      return new CMDOP(OP_WHERE, GetExpr0());
   }
   PCMDOP GetExpr0()
   {
      PCMDOP pExpr = GetExpr1();
      switch( PeekTokenType() ) {
      case OP_AND:
      case OP_OR:
         {
            TOKEN t = GetToken();
            return new CMDOP(t, pExpr, GetExpr0());
         }
      default:
         return pExpr;
      }
   }
   PCMDOP GetExpr1()
   {
      PCMDOP pExpr = GetExpr2();
      switch( PeekTokenType() ) {
      case OP_OPERATOR:
         {
            TOKEN t = GetToken();
            return new CMDOP(t, pExpr, GetExpr2());
         }
      default:
         return pExpr;
      }
   }
   PCMDOP GetExpr2()
   {
      PCMDOP pExpr = GetExpr3();
      switch( PeekTokenType() ) {
      case OP_ARITH:
         {
            TOKEN t = GetToken();
            return new CMDOP(t, pExpr, GetExpr2());
         }
      default:
         return pExpr;
      }
   }
   PCMDOP GetExpr3()
   {
      switch( PeekTokenType() ) {
      case OP_PAREN_L:
         {
            GetToken();
            PCMDOP pInnerExpr =  GetExpr0();
            if( GetToken().type!=OP_PAREN_R ) throw ERR_SYNTAX;
            return pInnerExpr;
         }
      default:
         return Primitive();
      }
   }
   PCMDOP Primitive()
   {
      switch( PeekTokenType() ) {
      case OP_KEYWORD:
      case OP_INTEGER:
      case OP_STRING:
         return new CMDOP(GetToken());
      default:
         return NULL;
      }
   }

   // Tokenizer

   TOKEN GetToken()
   {
      while( *m_pstr && *m_pstr<=_T(' ') ) m_pstr = ::CharNext(m_pstr);
      if( !*m_pstr ) return OP_END;
      switch( *m_pstr ) {
      case _T(','):
         m_pstr = ::CharNext(m_pstr);
         return TOKEN(OP_COMMA);
      case _T('('):
         m_pstr = ::CharNext(m_pstr);
         return TOKEN(OP_PAREN_L);
      case _T(')'):
         m_pstr = ::CharNext(m_pstr);
         return TOKEN(OP_PAREN_R);
      case _T('='):
      case _T('<'):
      case _T('>'):
         {
            TCHAR szOp[3] = { 0 };
            TCHAR c = szOp[0] = *m_pstr;
            m_pstr = ::CharNext(m_pstr);
            if( c!=_T('=') && *m_pstr==_T('=') ) {
               szOp[1] = _T('=');
               m_pstr = ::CharNext(m_pstr);
            }
            return TOKEN(OP_OPERATOR, szOp);
         }
      case _T('0'):
      case _T('1'):
      case _T('2'):
      case _T('3'):
      case _T('4'):
      case _T('5'):
      case _T('6'):
      case _T('7'):
      case _T('8'):
      case _T('9'):
         {
            TOKEN t(OP_INTEGER);
            LPTSTR p = t.value;
            while( *m_pstr && ::IsCharAlphaNumeric(*m_pstr) ) {
               *p = *m_pstr;
               p = ::CharNext(p);
               m_pstr = ::CharNext(m_pstr);
            }
            *p = _T('\0');
            return t;
         }
      case _T('+'):
      case _T('-'):
      case _T('*'):
      case _T('/'):
         {
            TOKEN t(OP_ARITH, *m_pstr);
            m_pstr = ::CharNext(m_pstr);
            return t;
         }
      case _T('\"'):
      case _T('\''):
         TCHAR chDelim = *m_pstr;
         m_pstr = ::CharNext(m_pstr); // skip quote
         TOKEN t(OP_STRING);
         LPTSTR p = t.value;
         int cCount = 0;
         while( *m_pstr && *m_pstr!=chDelim ) {
            *p = *m_pstr;
#ifdef _MBCS
            if( ::IsDBCSLeadByte(*p) ) *(p+1) = *(m_pstr+1);
#endif
            p = ::CharNext(p);
            m_pstr = ::CharNext(m_pstr);
            if( cCount++ >= MAX_TOKEN_STRLEN ) throw ERR_TOOLONG;
         }
         *p = _T('\0');
         if( *m_pstr!=chDelim ) throw ERR_DELIMTEREXPECTED;
         m_pstr = ::CharNext(m_pstr); // skip quote
         return t;
      }

      TOKEN t(OP_KEYWORD);
      LPTSTR p = t.value;
      int cCount = 0;
      while( *m_pstr && ::IsCharAlphaNumeric(*m_pstr) ) {
         *p = *m_pstr;
#ifdef _MBCS
         if( ::IsDBCSLeadByte(*p) ) *(p+1) = *(m_pstr+1);
#endif
         p = ::CharNext(p);
         m_pstr = ::CharNext(m_pstr);
         if( cCount++ >= MAX_TOKEN_STRLEN ) throw ERR_TOOLONG;
      }
      *p = _T('\0');

      // Match with known keywords
      if( _tcsicmp(t.value, _T("SELECT"))==0 ) return TOKEN(OP_SELECT);
      if( _tcsicmp(t.value, _T("UPDATE"))==0 ) return TOKEN(OP_UPDATE);
      if( _tcsicmp(t.value, _T("FROM"))==0 ) return TOKEN(OP_FROM);
      if( _tcsicmp(t.value, _T("WHERE"))==0 ) return TOKEN(OP_WHERE);
      if( _tcsicmp(t.value, _T("SET"))==0 ) return TOKEN(OP_SET);
      if( _tcsicmp(t.value, _T("AND"))==0 ) return TOKEN(OP_AND);
      if( _tcsicmp(t.value, _T("OR"))==0 ) return TOKEN(OP_OR);
      return t;
   }
   OPTYPE PeekTokenType()
   {
      // TODO: Optimize with a cache of last token!
      LPCTSTR pstrOld = m_pstr; // backup old text position
      TOKEN t = GetToken();
      m_pstr = pstrOld;
      return t.type;
   }
};


#endif // !defined(AFX_SQLPARSER_H__20010927_3552_9275_D2A6_0080AD509054__INCLUDED_)

