// NLSResource.cpp : Implementation of CResource

#include "stdafx.h"

#include "NLS.h"
#include "NLSResource.h"

#define TEXT_NA "n/a"
#define REPLACEMENT_CHAR '$'
#define MAX_STR_LENGTH 1024


/////////////////////////////////////////////////////////////////////////////
// CResource

BOOL CResource::_LoadResourceBitmap(HINSTANCE hInstance, 
                                    TCHAR *pszString, 
                                    HBITMAP &hBitmap,
                                    HPALETTE &hPalette)
{
// Because ::LoadBitmap() doesn't return the HPALETTE we need
// to load the resource ourself.
// Loading as a DIB keeps a 256 colour palette intact.
   HRSRC  hRsrc;
   if( hRsrc = ::FindResource(hInstance, pszString, RT_BITMAP) ) {
      HGLOBAL hGlobal = ::LoadResource(hInstance, hRsrc);
      LPBITMAPINFOHEADER lpbi = (LPBITMAPINFOHEADER)::LockResource(hGlobal);

      HDC hdc = ::GetDC(NULL);
      int iNumColors;
      hPalette =  _CreateDIBPalette((LPBITMAPINFO)lpbi, &iNumColors);
      if( hPalette!=NULL ) {
         ::SelectPalette(hdc, hPalette, FALSE);
         ::RealizePalette(hdc);
      }
      hBitmap = ::CreateDIBitmap(hdc, (LPBITMAPINFOHEADER)lpbi,
                        (LONG)CBM_INIT, (LPSTR)lpbi + lpbi->biSize + iNumColors * sizeof(RGBQUAD),
                        (LPBITMAPINFO)lpbi, DIB_RGB_COLORS );

      ::ReleaseDC(NULL,hdc);
      UnlockResource(hGlobal);
      ::FreeResource(hGlobal);
   }
   return hBitmap!=NULL;
} 

HPALETTE CResource::_CreateDIBPalette(LPBITMAPINFO lpbmi, LPINT lpiNumColors)
{
   ATLASSERT(lpbmi);
   ATLASSERT(lpiNumColors);

   HPALETTE hPal = NULL;

   LPBITMAPINFOHEADER lpbi = (LPBITMAPINFOHEADER)lpbmi;
   if (lpbi->biBitCount <= 8)
      *lpiNumColors = (1 << lpbi->biBitCount);
   else
      *lpiNumColors = 0;  // No palette needed for 24 BPP DIB
   if( lpbi->biClrUsed > 0) *lpiNumColors = lpbi->biClrUsed;  // Use biClrUsed
   if( *lpiNumColors!=0 ) {
      HANDLE hLogPal = ::GlobalAlloc(GHND, sizeof(LOGPALETTE) +
                             sizeof(PALETTEENTRY) * (*lpiNumColors));
      LPLOGPALETTE lpPal = (LPLOGPALETTE)::GlobalLock(hLogPal);
      lpPal->palVersion    = 0x300;
      lpPal->palNumEntries = *lpiNumColors;

      for( int i=0;  i<*lpiNumColors;  i++ ) {
         lpPal->palPalEntry[i].peRed   = lpbmi->bmiColors[i].rgbRed;
         lpPal->palPalEntry[i].peGreen = lpbmi->bmiColors[i].rgbGreen;
         lpPal->palPalEntry[i].peBlue  = lpbmi->bmiColors[i].rgbBlue;
         lpPal->palPalEntry[i].peFlags = 0;
      }
      hPal = ::CreatePalette(lpPal);
      ::GlobalUnlock(hLogPal);
      ::GlobalFree(hLogPal);
   }
   return hPal;
} 


/////////////////////////////////////////////////////////////////////////////
// IResource

STDMETHODIMP CResource::Init(BSTR Filename)
{
   USES_CONVERSION;
   if( Filename==NULL ) return E_POINTER;
   //
   if( m_hInstance!=NULL ) Close();
   m_hInstance = ::LoadLibraryEx(W2CT(Filename), NULL, LOAD_LIBRARY_AS_DATAFILE);
   if( m_hInstance==NULL ) return Error(IDS_ERR_LOADRES);
   m_bstrFilename = Filename;
   return S_OK;
}

STDMETHODIMP CResource::Close()
{
   if( m_hInstance==NULL ) return S_OK;
   //
   ::FreeLibrary(m_hInstance);
   m_hInstance = NULL;
   m_bstrFilename.Empty();
   return S_OK;
}

STDMETHODIMP CResource::FormatResString(long ID, SAFEARRAY** ArgList, BSTR *pVal)
{
   ATLASSERT(ArgList);
   ATLASSERT(pVal);
   ATLASSERT(m_hInstance);
   if( pVal==NULL ) return E_POINTER;
   *pVal = NULL;
   if( ID==0 || ID==1 ) return E_INVALIDARG;
   if( m_hInstance==NULL ) return Error(IDS_ERR_NORESLOADED);

   // Load string from resource
   USES_CONVERSION;
   int nMax = MAX_STR_LENGTH;
   LPTSTR pstrBuffer = (LPTSTR)alloca( nMax ); // allocate space on stack
   if( pstrBuffer==NULL ) return E_OUTOFMEMORY;
   int res = ::LoadString( m_hInstance, ID, pstrBuffer, nMax );
   if( res==0 ) {
      CComBSTR bstr(_T(TEXT_NA));
      *pVal = bstr.Detach();
      return S_OK;
   }

   // Parse argument list
   if( ArgList!=NULL ) {
      long lLBound, lUBound;
      HRESULT hr;
      hr = ::SafeArrayGetLBound(*ArgList, 1, &lLBound);
      if( FAILED(hr) ) return hr;
      hr = ::SafeArrayGetUBound(*ArgList, 1, &lUBound);
      if( FAILED(hr) ) return hr;
      VARIANT *pvData;
      hr = ::SafeArrayAccessData(*ArgList, (VOID **)&pvData);
      if( FAILED(hr) ) return hr;

      for( int i=0; i< lUBound-lLBound+1; i++ ) {
         TCHAR szItem[6];
         ::wsprintf(szItem, _T("%c%i"), REPLACEMENT_CHAR, i+1);
         LPCTSTR pstrReplacement = OLE2CT(pvData[ lLBound+i ].bstrVal);
         LPTSTR pstrTarget;
         int nTokenLen = _tcslen(szItem);
         int nReplacementLen = _tcslen(pstrReplacement);
         int nOldLength = nMax - _tcslen(pstrBuffer);

         while( (pstrTarget = _strstr(pstrBuffer, szItem)) != NULL ) {
            int nBalance = nOldLength - ((int)(DWORD_PTR)(pstrTarget - pstrBuffer) + nTokenLen);
            memmove( pstrTarget + nReplacementLen, pstrTarget + nTokenLen,  (nBalance+1) * sizeof(TCHAR));
            memcpy(pstrTarget, pstrReplacement, nReplacementLen * sizeof(TCHAR));
            nOldLength += (nReplacementLen - nTokenLen);
         }

      }
      ::SafeArrayUnaccessData(*ArgList);
   };

   CComBSTR bstr( pstrBuffer );
   *pVal = bstr.Detach();
   return S_OK;
}

/**
 * Loads a string from a resource file and returns it as a property of a control.
 *
 * If the resource string is not found the default value is returned (if defined).
 * If no default value is passed, the string "n/a" is returned.
 *
 * @param ID The Resouce ID to locate.
 * @param Default The optional default value if the string is not found.
 * @return The localized string.
 */
STDMETHODIMP CResource::LoadResString(long ID, BSTR Default, BSTR *pVal)
{
   ATLASSERT(m_hInstance);
   ATLASSERT(pVal);
   if( pVal==NULL ) return E_POINTER;
   *pVal = NULL;
   if( ID==0 || ID==1 ) return E_INVALIDARG;
   if( m_hInstance==NULL ) return Error(IDS_ERR_NORESLOADED);
   //
   USES_CONVERSION;
   int nMax = MAX_STR_LENGTH;
   LPTSTR pstrBuffer = (LPTSTR)alloca( nMax ); // allocate space on stack
   if( pstrBuffer==NULL ) return E_OUTOFMEMORY;
   int res = ::LoadString( m_hInstance, ID, pstrBuffer, nMax );
   if (res==0) {
      _tcscpy( pstrBuffer, Default!=NULL ? OLE2T(Default) : _T(TEXT_NA) );
   }
   CComBSTR bstr( pstrBuffer );
   *pVal = bstr.Detach();
   return S_OK;
};

STDMETHODIMP CResource::LoadResPicture(long ID, short Type, IUnknown **pVal)
{
   ATLASSERT(m_hInstance);
   ATLASSERT(pVal);
   if( pVal==NULL ) return E_POINTER;
   *pVal = NULL;
   if( m_hInstance==NULL ) return Error(IDS_ERR_NORESLOADED);
   //
   BOOL bTransferOwnership = TRUE;
   PICTDESC pdesc = { 0 };
   pdesc.cbSizeofstruct = sizeof(pdesc);
   switch( Type ) {
   case 0: // vbResBitmap
      pdesc.picType = PICTYPE_BITMAP;
      _LoadResourceBitmap(m_hInstance, MAKEINTRESOURCE(ID), pdesc.bmp.hbitmap, pdesc.bmp.hpal);
      //pdesc.bmp.hbitmap = ::LoadBitmap(m_hInstance, MAKEINTRESOURCE(ID));
      //pdesc.bmp.hpal = NULL;
      break;
   case 1: // vbResIcon
      pdesc.picType = PICTYPE_ICON;
      pdesc.icon.hicon = ::LoadIcon(m_hInstance, MAKEINTRESOURCE(ID));
      break;
   case 2: // vbResCursor
      pdesc.picType = PICTYPE_ICON;
      pdesc.icon.hicon = ::LoadCursor(m_hInstance, MAKEINTRESOURCE(ID));
      break;
   default:
      return E_INVALIDARG;
   }
   HRESULT hr;
   LPPICTURE pPict = NULL;
   hr = ::OleCreatePictureIndirect(&pdesc, IID_IPicture, bTransferOwnership, (LPVOID*)&pPict);
   if( SUCCEEDED(hr) ) {
      LPPICTUREDISP pPictDisp = NULL;
      hr = pPict->QueryInterface(IID_IPictureDisp, (LPVOID*)&pPictDisp);
      *pVal = pPictDisp;
   }
   return hr;
}

STDMETHODIMP CResource::get_Filename(BSTR *pVal)
{
   ATLASSERT(m_hInstance);
   ATLASSERT(pVal);
   if( pVal==NULL ) return E_POINTER;
   return m_bstrFilename.CopyTo(pVal);
}

STDMETHODIMP CResource::get_hInstance(long *pVal)
{
   ATLASSERT(pVal);
   if( pVal==NULL ) return E_POINTER;
   if( m_hInstance==NULL ) return Error(IDS_ERR_NORESLOADED);
   //
   *pVal = (long)m_hInstance;
   return S_OK;
}

/**
 * This method attempts to locate a Resource DLL and returns the full
 * path to it if found.
 *
 * The Resource DLL must have the following format:
 * <pre>Path\Title.xxx</pre>
 * where xxx is language abbreviation for the Locale (mostly based on the ISO Standard 3166).
 * E.g. for English/US the 3 letter code is "enu", danish is "dan" etc.
 *
 * The method will first look for a Resource DLL that matches the Locale
 * passed in the LCID argument.
 * If no DLL is found, the current System and User Locales will be searched
 * for.<br>
 * If none of the above searches comes up with a DLL, the method will look
 * for an English/USA (extension ENU) file.<br>
 * If a Resource DLL cannot be found at all, the method fails.<br>
 *
 * @param LCID The Locale to give first priority in the search.
 *             If this argument is 0 then the System Locale is preferred.
 * @param Path The path to look for the Resource DLL.
 * @param Title The filename title of the Resource DLL.
 * @return The full path to a Resource DLL.
 */
STDMETHODIMP CResource::LocateResource(long LCID, BSTR Path, BSTR Title, BSTR *pVal)
{
   ATLASSERT(pVal);
   if( pVal==NULL ) return E_POINTER;
   if( Path==NULL ) return E_INVALIDARG;
   if( Title==NULL ) return E_INVALIDARG;
   //
   USES_CONVERSION;
   TCHAR szFilename[MAX_PATH], szBuffer[MAX_PATH], szLocaleName[10];
   // First get path
   _tcscpy( szFilename, OLE2CT(Path) );
   // Append \-char
   _AppendCharIfMissing(szFilename,_T('\\'));
   // Then add filetitle
   _tcscat( szFilename, OLE2CT(Title) );
   // Make sure it has the .-char extension
   _AppendCharIfMissing(szFilename,_T('.'));
   // Ready to locate resource dll
   BOOL ok = FALSE;
   // Try with hardcoded locale
   if( !ok ) {
      if( (LCID!=0) && ::IsValidLocale(LCID,LCID_INSTALLED) ) {
         ::GetLocaleInfo( LCID, LOCALE_SABBREVLANGNAME, szLocaleName, sizeof(szLocaleName)/sizeof(TCHAR) );
         ::wsprintf( szBuffer, _T("%s%s"), szFilename, szLocaleName );
         ok = _FileExists(szBuffer);
      };
   };
   // Try with system default
   if( !ok ) {
      LCID = ::GetSystemDefaultLCID();
      if( (LCID!=0) && ::IsValidLocale(LCID,LCID_INSTALLED) ) {
         ::GetLocaleInfo( LCID, LOCALE_SABBREVLANGNAME, szLocaleName, sizeof(szLocaleName)/sizeof(TCHAR) );
         ::wsprintf( szBuffer, _T("%s%s"), szFilename, szLocaleName );
         ok = _FileExists(szBuffer);
      };
   };
   // Try with user default
   if( !ok ) {
      LCID = ::GetUserDefaultLCID();
      if( (LCID!=0) && ::IsValidLocale(LCID,LCID_INSTALLED) ) {
         ::GetLocaleInfo( LCID, LOCALE_SABBREVLANGNAME, szLocaleName, sizeof(szLocaleName)/sizeof(TCHAR) );
         ::wsprintf( szBuffer, _T("%s%s"), szFilename, szLocaleName );
         ok = _FileExists(szBuffer);
      };
   };
   // Finally hardcode to English-USA (should always be present and valid)
   if( !ok ) {
      ::wsprintf( szBuffer, _T("%senu"), szFilename );
      ok = _FileExists(szBuffer);
   };
   if( !ok ) return Error(IDS_ERR_RESNOTFOUND);
   CComBSTR bstr( szBuffer );
   *pVal = bstr.Detach();
   return S_OK;
}

