// Info.cpp : Implementation of CInfo

#include "stdafx.h"

#include "NLS.h"
#include "NLSLocale.h"


/////////////////////////////////////////////////////////////////////////////
// CLocale

/**
 * Returns the current Locale.
 *
 * Upen create, the value is set to the Thread Locale.
 *
 * @return The current LCID.
 */
STDMETHODIMP CLocale::get_LCID(long *pVal)
{
   ATLASSERT(pVal);
   if( pVal==NULL ) return E_POINTER;
   //
   *pVal = m_lcid;
   return S_OK;
}

/**
 * Set the current Locale.
 *
 * To search for locale IDs, lookup "Appendix F Locale-Specific Code Page Information"
 * on MSDN.
 */
STDMETHODIMP CLocale::put_LCID(long newVal)
{
   if( !::IsValidLocale(newVal, LCID_INSTALLED ) ) return Error(IDS_ERR_LCIDINVALID);
   m_lcid = newVal;
   return S_OK;
}

/**
 * Returs TRUE if the locale is both supported by Windows and currently installed.
 *
 * @param LCID Locale Identifier to check.
 * @return Flag indicating if the Locale is supported by the system.
 */
STDMETHODIMP CLocale::IsValidLocale(long LCID, VARIANT_BOOL *pRes)
{
   BOOL bRes = ::IsValidLocale( LCID, LCID_INSTALLED );
   if( pRes!=NULL ) *pRes = bRes ? VARIANT_TRUE : VARIANT_FALSE;
   return bRes ? S_OK : S_FALSE;
}

/**
 * Assigns the locale to the current thread.
 *
 * Windows supports setting a locale on thread-basis.
 */
STDMETHODIMP CLocale::SetThreadLocale()
{
   BOOL bRes = ::SetThreadLocale(m_lcid);
   return bRes ? S_OK : HRESULT_FROM_WIN32(::GetLastError());
}

/**
 * Returns local information.
 *
 * @param Option Locale information to request.
 * @return Local info in string format.
 */
STDMETHODIMP CLocale::GetInfo(long Option, BSTR *pVal)
{
   ATLASSERT(pVal);
   if( pVal==NULL ) return E_POINTER;
   //
   TCHAR sBuffer[128] = { 0 };
   int ret = ::GetLocaleInfo( m_lcid, Option, sBuffer, sizeof(sBuffer)/sizeof(TCHAR) );
   if( ret==0 ) return HRESULT_FROM_WIN32(::GetLastError());
   CComBSTR bstr = sBuffer;
   *pVal = bstr.Detach();
   return S_OK;
}

/**
 * Returns the maximum length of any character of the locale
 */
STDMETHODIMP CLocale::get_MaxCharLength(long *pVal)
{
   ATLASSERT(pVal);
   if( pVal==NULL ) return E_POINTER;
   //
   CPINFO cpinfo;
   TCHAR sBuffer[16];
   UINT cp;
#ifndef LOCALE_IDEFAULTOEMCODEPAGE
   ::GetLocaleInfo( m_lcid, LOCALE_IDEFAULTCODEPAGE, sBuffer, sizeof(sBuffer)/sizeof(TCHAR) );
#else
   ::GetLocaleInfo( m_lcid, LOCALE_IDEFAULTOEMCODEPAGE, sBuffer, sizeof(sBuffer)/sizeof(TCHAR) );
#endif
   cp = _ttoi( sBuffer );
   if( ::GetCPInfo( cp, &cpinfo )==FALSE ) return E_FAIL;
   *pVal = cpinfo.MaxCharSize;
   return S_OK;
}

/**
 * Returns the langauge name of the current locale.
 */
STDMETHODIMP CLocale::get_Language(BSTR *pVal)
{
   ATLASSERT(pVal);
   return GetInfo(LOCALE_SLANGUAGE, pVal);
}
