
#include "stdafx.h"

//////////////////////////////////////
//

HRESULT CEngine::Init()
{
   HRESULT Hr;

   if( FAILED(Hr = CreateContext()) ) return Hr;
   if( FAILED(Hr = InitView()) ) return Hr;
   if( FAILED(Hr = InitStage()) ) return Hr;

   return S_OK;
};

HRESULT CEngine::Close()
{
   ReleaseContext();
   return S_OK;
};

HRESULT CEngine::Draw()
{
   HRESULT Hr;
   int i;

   if( !m_bReady ) return E_FAIL;
  
   if( _System.m_bActive ) {

      //
      // Process keyboard input
      //
      D3DXVECTOR3 vecT(0.0f, 0.0f, 0.0f);
      D3DXVECTOR3 vecR(0.0f, 0.0f, 0.0f);

      // Call PlayerMove() for all Nodes
      for( int i=0; i<_System.m_nSpectators; i++ ) _System.m_ppSpectators[i]->PlayerMove(vecT,vecR);

      m_vecVelocity = m_vecVelocity * 0.9f + vecT * 0.1f;
      m_vecAngularVelocity = m_vecAngularVelocity * 0.9f + vecR * 0.1f;

      //
      // Update position and view matricies
      //
      D3DXMATRIX matT, matR;
      D3DXQUATERNION qR;

      float fSec = m_timer.GetSecsPerFrame();
      vecT = m_vecVelocity * fSec * m_fSpeed;
      vecR = m_vecAngularVelocity * fSec * m_fAngularSpeed;

      D3DXMatrixTranslation(&matT, vecT.x, vecT.y, vecT.z);
      D3DXMatrixMultiply(&m_matPosition, &matT, &m_matPosition);

      D3DXQuaternionRotationYawPitchRoll(&qR, vecR.y, vecR.x, vecR.z);
      D3DXMatrixRotationQuaternion(&matR, &qR);
      D3DXMatrixMultiply(&m_matPosition, &matR, &m_matPosition);

      if(m_matPosition.m31 < 1.0f) m_matPosition.m31 = 1.0f;

      D3DXMatrixInverse(&m_matView, NULL, &m_matPosition);
   }

   // Call PreDraw for all Nodes
   for( i=0; i<_System.m_nNodes; i++ ) _System.m_ppNodes[i]->PreDraw();

   //
   // Begin Scene
   //
   if( SUCCEEDED(m_pD3DDevice->BeginScene()) ) {
      m_pD3DX->Clear(D3DCLEAR_TARGET | D3DCLEAR_ZBUFFER);

      m_pD3DDevice->SetTransform(D3DTRANSFORMSTATE_VIEW, m_matView);
      m_pD3DDevice->SetTransform(D3DTRANSFORMSTATE_WORLD, m_matIdentity);

      // Draw all nodes
      for( int i=0; i<_System.m_nNodes; i++ ) _System.m_ppNodes[i]->Draw();

      // End Scene
      m_pD3DDevice->EndScene();
    }

#ifdef _DEBUG
   //
   // Draw frame rate
   //
   DWORD dwColor;
   dwColor = 0xff7fcf7f;
   CHAR sz[32];
   sprintf(sz, "%d fps", (int)m_timer.GetFramesPerSec());
   m_pD3DX->DrawDebugText(0.01f, 0.0f, dwColor, sz);
#endif

   //
   // Update frame
   //
   Hr = m_pD3DX->UpdateFrame(0);
   if( DDERR_SURFACELOST == Hr || DDERR_SURFACEBUSY == Hr ) {
      Hr = m_pDD->TestCooperativeLevel();
      if( SUCCEEDED(Hr) ) {
         if( FAILED(Hr = RestoreContext()) ) return Hr;
      }
      else if( DDERR_WRONGMODE == Hr ) {
         if( FAILED(Hr = ReleaseContext()) ) return Hr;
         if( FAILED(Hr = CreateContext()) ) return Hr;
      }
   }

   // Call PostDraw for all Nodes
   for( i=0; i<_System.m_nNodes; i++ ) _System.m_ppNodes[i]->PostDraw();

   if(_System.m_bActive) m_timer.Frame();

   return S_OK;
};


//////////////////////////////////////
//
// Context
//

HRESULT CEngine::CreateContext()
{
   HRESULT Hr;
   _System.LogText("Creating Engine context...");

   //
   // Initialize D3DX
   //
   if( FAILED(Hr = D3DXInitialize()) ) return _System.LogError(Hr);

   if( _System.m_bFullScreen ) {
     Hr = D3DXCreateContextEx(m_dwHwLevel, 
        D3DX_CONTEXT_FULLSCREEN,  
        _System.m_hwnd, 
        NULL, 
        D3DX_DEFAULT, 
        0,
        D3DX_DEFAULT, 
        0, 
        m_dwNumBackBuffers, 
        640, 480, 
        D3DX_DEFAULT, 
        &m_pD3DX);
   }
   else {
     Hr = D3DXCreateContextEx(m_dwHwLevel, 
        0, 
        _System.m_hwnd, 
        NULL, 
        D3DX_DEFAULT, 
        0,
        D3DX_DEFAULT, 
        0, 
        m_dwNumBackBuffers, 
        D3DX_DEFAULT, D3DX_DEFAULT, 
        D3DX_DEFAULT, 
        &m_pD3DX);
   }
   if( FAILED(Hr) ) return _System.LogError(Hr);
   if( (m_pDD = m_pD3DX->GetDD())==NULL ) return _System.LogError(Hr);
   if( (m_pD3D = m_pD3DX->GetD3D())==NULL ) return _System.LogError(Hr);
   if( (m_pD3DDevice = m_pD3DX->GetD3DDevice())==NULL ) return _System.LogError(Hr);

   //
   // Grok device caps
   //
   ZeroMemory(&m_D3DDeviceDesc, sizeof(D3DDEVICEDESC7));
   m_pD3DDevice->GetCaps(&m_D3DDeviceDesc);

#ifdef _DEBUG
   _System.LogText("Device Capabilities:");
   _System.LogText("  Depth: %d", m_D3DDeviceDesc.dwDeviceRenderBitDepth);
#endif

   D3DXMatrixIdentity(&m_matIdentity);
   return S_OK;
};

HRESULT CEngine::RestoreContext()
{
   _System.LogText("Restoring Engine context...");
   HRESULT hr;   
   if( FAILED( hr = m_pD3DX->RestoreSurfaces() ) ) return hr;

   // Call RestoreSurfaces() on all nodes
   for( int i=0; i<_System.m_nNodes; i++ ) _System.m_ppNodes[i]->RestoreSurfaces();

   return S_OK;
}

HRESULT CEngine::ReleaseContext()
{
   _System.LogText("Releasing Engine context...");
   m_bReady = false;

   // Call ReleaseSurfaces() on all nodes
   for( int i=0; i<_System.m_nNodes; i++ ) _System.m_ppNodes[i]->ReleaseSurfaces();

   RELEASE(m_pD3DDevice);
   RELEASE(m_pD3D);
   RELEASE(m_pDD);
   RELEASE(m_pD3DX);

   D3DXUninitialize();

   return S_OK;
}



////////////////////////////////
//
// Context
//

HRESULT CEngine::InitView()
{
   HRESULT Hr;

   D3DXMatrixTranslation(&m_matView, 0.0f, 0.0f, -20.0f);
   D3DXMatrixTranslation(&m_matPosition, 0.0f, 0.0f, 20.0f);
   D3DXMatrixPerspectiveFov(&m_matProjection, D3DXToRadian(60.0f), 3.0f / 4.0f, 0.1f, 100.0f);

   Hr = GetRGB16();
   if( FAILED(Hr) ) return Hr;

   m_timer.Start(30.0f);

   return S_OK;
};

HRESULT CEngine::InitStage()
{
   //
   // Setup render states
   //
   m_pD3DX->SetClearColor(m_colorClear);

   m_pD3DDevice->SetTransform(D3DTRANSFORMSTATE_VIEW, m_matView);
   m_pD3DDevice->SetTransform(D3DTRANSFORMSTATE_WORLD, m_matIdentity);
   m_pD3DDevice->SetTransform(D3DTRANSFORMSTATE_PROJECTION, m_matProjection);

   float fFogStart = 0.0f, fFogEnd = 40.0f;
   m_pD3DDevice->SetRenderState(D3DRENDERSTATE_FOGCOLOR, m_colorClear);
   m_pD3DDevice->SetRenderState(D3DRENDERSTATE_FOGTABLEMODE, D3DFOG_LINEAR);
   m_pD3DDevice->SetRenderState(D3DRENDERSTATE_FOGSTART, *((DWORD *) &fFogStart));
   m_pD3DDevice->SetRenderState(D3DRENDERSTATE_FOGEND, *((DWORD *) &fFogEnd));

   m_pD3DDevice->SetRenderState(D3DRENDERSTATE_CULLMODE, D3DCULL_CCW);

   m_pD3DDevice->SetRenderState(D3DRENDERSTATE_FILLMODE, D3DFILL_SOLID);
   m_pD3DDevice->SetRenderState(D3DRENDERSTATE_SHADEMODE, D3DSHADE_GOURAUD);
   m_pD3DDevice->SetRenderState(D3DRENDERSTATE_DITHERENABLE, TRUE);

   m_pD3DDevice->SetTextureStageState(0, D3DTSS_MINFILTER, D3DTFN_LINEAR);
   m_pD3DDevice->SetTextureStageState(0, D3DTSS_MAGFILTER, D3DTFG_LINEAR);
   m_pD3DDevice->SetTextureStageState(0, D3DTSS_MIPFILTER, D3DTFP_POINT);
   m_pD3DDevice->SetTextureStageState(0, D3DTSS_COLOROP, D3DTOP_MODULATE);
   m_pD3DDevice->SetTextureStageState(0, D3DTSS_ALPHAOP, D3DTOP_MODULATE);


   //
   // Create lights
   //
   for( int i=0; i<_System.m_nLights; i++ ) {
      _System.m_ppLights[i]->Create();
   };
   m_pD3DDevice->SetRenderState(D3DRENDERSTATE_AMBIENT, 0xffffffff);

   //
   // Create material
   //
   D3DMATERIAL7 d3dMaterial;
   ZeroMemory(&d3dMaterial, sizeof(d3dMaterial));
   d3dMaterial.dcvDiffuse.r = 0.75f;
   d3dMaterial.dcvDiffuse.g = 0.75f;
   d3dMaterial.dcvDiffuse.b = 0.75f;
   d3dMaterial.dcvDiffuse.a = 0.25f;
   D3DXCOLOR colorLight(1.0f, 0.95f, 0.8f, 1.0f);
   d3dMaterial.dcvAmbient.r = colorLight.r;
   d3dMaterial.dcvAmbient.g = colorLight.g;
   d3dMaterial.dcvAmbient.b = colorLight.b;
   d3dMaterial.dcvSpecular.r = 0.5f;
   d3dMaterial.dcvSpecular.g = 0.5f;
   d3dMaterial.dcvSpecular.b = 0.5f;
   d3dMaterial.dcvSpecular.b = 0.5f;
   d3dMaterial.dvPower = 10.0f;
   m_pD3DDevice->SetMaterial(&d3dMaterial);

   return S_OK;
};

////////////////////////////////
//
// Textures
//

HRESULT CEngine::CreateTexture(LPCSTR szFilename, short &ID)
{
   HRESULT Hr;
   CHAR szTexName[MAX_PATH];
   strcpy( szTexName, _System.m_szResPath );
   strcat( szTexName, szFilename );
   ID = m_nTextures;
   D3DX_SURFACEFORMAT sf = D3DX_SF_UNKNOWN;
   if(FAILED(Hr = D3DXCreateTextureFromFile(m_pD3DDevice, 0, 0, 0, &sf, NULL, &m_ppTextures[m_nTextures], NULL, szTexName, D3DX_FT_LINEAR))) {
      m_ppTextures[m_nTextures] = NULL;
      _System.LogError("Unable to load texture: '%s' (%X08)", szFilename, Hr);
      return E_FAIL;
   }
   else {
      m_nTextures++;
      return S_OK;
   };
};

HRESULT CEngine::CreateTexture(LPDIRECTDRAWSURFACE7 pSurface, short &ID)
{
   if( pSurface==NULL ) return E_FAIL;
   m_ppTextures[m_nTextures] = pSurface;
   ID = m_nTextures++;
   return S_OK;
};

HRESULT CEngine::GetRGB16()
{
   LPDIRECTDRAWSURFACE7 pSurface;
   pSurface = m_pD3DX->GetPrimary();
   if( pSurface==NULL ) return E_FAIL;

   DDSURFACEDESC2 ddsd;
   BYTE shiftcount;
   
   // Get a surface despriction
   ddsd.dwSize = sizeof(ddsd);
   ddsd.dwFlags = DDSD_PIXELFORMAT;
   if(pSurface->GetSurfaceDesc(&ddsd) != DD_OK) return E_UNEXPECTED;

   // Fill in the masking values for extracting colors
   m_rgb16.Mask.rgbRed = ddsd.ddpfPixelFormat.dwRBitMask;
   m_rgb16.Mask.rgbGreen = ddsd.ddpfPixelFormat.dwGBitMask;
   m_rgb16.Mask.rgbBlue = ddsd.ddpfPixelFormat.dwBBitMask;

   // Get red
   shiftcount = 0;
   while(!(ddsd.ddpfPixelFormat.dwRBitMask & 1))
   {
      ddsd.ddpfPixelFormat.dwRBitMask >>= 1;
      shiftcount++;
   }
   m_rgb16.depth.rgbRed = (BYTE)ddsd.ddpfPixelFormat.dwRBitMask;
   m_rgb16.Position.rgbRed = shiftcount;
   m_rgb16.Amount.rgbRed = (ddsd.ddpfPixelFormat.dwRBitMask == 0x1f) ? 3 : 2;

   // Get green
   shiftcount = 0;
   while(!(ddsd.ddpfPixelFormat.dwGBitMask & 1))
   {
      ddsd.ddpfPixelFormat.dwGBitMask >>= 1;
      shiftcount++;
   }
   m_rgb16.depth.rgbGreen = (BYTE)ddsd.ddpfPixelFormat.dwGBitMask;
   m_rgb16.Position.rgbGreen = shiftcount;
   m_rgb16.Amount.rgbGreen = (ddsd.ddpfPixelFormat.dwGBitMask == 0x1f) ? 3 : 2;
    
   // Get blue
   shiftcount = 0;
   while(!(ddsd.ddpfPixelFormat.dwBBitMask & 1))
   {
      ddsd.ddpfPixelFormat.dwBBitMask >>= 1;
      shiftcount++;
   }
   m_rgb16.depth.rgbBlue = (BYTE)ddsd.ddpfPixelFormat.dwBBitMask;
   m_rgb16.Position.rgbBlue = shiftcount;
   m_rgb16.Amount.rgbBlue = (ddsd.ddpfPixelFormat.dwBBitMask == 0x1f) ? 3 : 2;
    
   pSurface->Release();
   return S_OK;
}

