// RegValues.cpp : Implementation of CRegValues

#include "stdafx.h"

#include "EasyReg.h"
#include "RegValues.h"
#include "RegValue.h"
#include "enum.h"

/**
 * <B>RegValues</B> implements a collection of Registry Values.  Registry Values are always attached to a
 * Registry Node.
 * <P>
 * The <B>RegValues</B> collection has methods to add a new value and remove a value, and list all the attached values.
 * It also has the usual methods and properties for a collection: <B>Count</B>, <B>_NewEnum</B>, and <B>Item</B>.
 */
/* INTERFACE IRegValues */


/////////////////////////////////////////////////////////////////////////////
// CRegValues


/////////////////////////////////////////////////////////////////////////////
// Implementation

HRESULT CRegValues::LocateValue(CRegKey &reg, VARIANT Index, LPTSTR szItem, DWORD cchItem)
// Lookup and open a registy node, then make sure that
// the Registry value exists...
{
  ATLASSERT(szItem);
  ATLASSERT(cchItem>0);
  USES_CONVERSION;
  switch( V_VT(&Index) ) {
  case VT_I1:
  case VT_I2:
    ::VariantChangeType(&Index, &Index, 0, VT_I4);
    // Fall through...
  case VT_I4:
    // Lookup index (position). Passed values are 1-based, but Registry index is 0-based
    if( ::RegEnumValue(reg, V_I4(&Index)-1, szItem, &cchItem, NULL, NULL, NULL, NULL) != ERROR_SUCCESS ) 
      return ATLERROR(RegValues, L"Invalid item index.");
    break;
  case VT_BSTR:
    ::lstrcpy(szItem, W2T(V_BSTR(&Index)));
    break;
  default:
    return ATLERROR(RegValues, L"Invalid argument.");
  };
  // Ok, do a simple check to see if the entry is present.
  // This will open the CRegKey object
  if( reg.Open(m_hkeyRoot, W2CT(m_sBranch), KEY_READ) != ERROR_SUCCESS )
    return ATLERROR(RegValues, L"Unable to open branch.");
  if( ::RegQueryValueEx(reg, szItem, NULL, NULL, NULL, NULL) != ERROR_SUCCESS )
    return ATLERROR(RegValues, L"Item not found.");
  return S_OK;
};

HRESULT CRegValues::GetValue(CRegKey &reg, LPCTSTR szItem, CComVariant &Result)
// Extract the Registry value. The result is put into a VARIANT according to
// the data type.
{
  ATLASSERT(szItem);
  DWORD dwType;
  DWORD dwSize;
  long res;
  // First we need to know the size and data type of the entry
  ::RegQueryValueEx(reg, szItem, NULL, &dwType, NULL, &dwSize );
  // Then we can read the data and put it into an appropriate VARIANT
  switch( dwType ) {
  case REG_DWORD:
    Result.vt = VT_I4;
    dwSize = sizeof(DWORD);
    res = ::RegQueryValueEx(reg, szItem, NULL, NULL, (LPBYTE)&(Result.lVal), &dwSize );  
    break;
  case REG_SZ:
  case REG_EXPAND_SZ: // Uh, might loose datatype on writes!!!
    {
      TCHAR szData[MAXVALUELEN];
      dwSize = sizeof(szData)/sizeof(TCHAR);
      res = ::RegQueryValueEx(reg, szItem, NULL, NULL, (LPBYTE)szData, &dwSize );  
      Result = szData;
    };
    break;
  default:
    // We only support a limited number of data types yet...
    return ATLERROR(RegValues, L"Value has unsupported data type.");
  };
  if( res!=ERROR_SUCCESS ) return ATLERROR(RegValues, L"Unable to query value contents.");
  return S_OK;
};

HRESULT CRegValues::InitValue(CRegValue *pObj, HKEY hkeyRoot, BSTR sBranch, LPCTSTR szName, CComVariant &vValue)
// Helper function to initialize a newly create CRegValue object
{
  ATLASSERT(pObj);
  if( pObj==NULL ) return E_POINTER;
  pObj->m_hkeyRoot = hkeyRoot;
  pObj->m_sBranch = sBranch;
  pObj->m_sName = szName;
  pObj->m_vValue = vValue;
  return S_OK;
};


/////////////////////////////////////////////////////////////////////////////
// IRegValues

STDMETHODIMP CRegValues::get_Item(VARIANT Index, IRegValue** ppvObject)
{
  ATLASSERT(ppvObject);
  if( ppvObject==NULL ) return E_POINTER;

  CRegKey reg;
  TCHAR szItem[MAXVALUELEN];
  HRESULT Hr;
  if( FAILED( Hr = LocateValue(reg, Index, szItem, sizeof(szItem)/sizeof(TCHAR)) ) ) return Hr;
  CComVariant vValue;
  if( FAILED( Hr = GetValue(reg, szItem, vValue) ) ) return Hr;
  reg.Close();

  CComObject<CRegValue>* pRegValue;
  Hr = CComObject<CRegValue>::CreateInstance(&pRegValue);
  if( FAILED(Hr) ) return Hr;
  InitValue(pRegValue, m_hkeyRoot, m_sBranch, szItem, vValue);
 
  return pRegValue->QueryInterface(ppvObject);
};

STDMETHODIMP CRegValues::get_Count(long* pRetVal)
{
  ATLASSERT(pRetVal);
  if( pRetVal==NULL ) return E_POINTER;

  DWORD dwCount;
  CRegKey reg;
  USES_CONVERSION;
  if( reg.Open(m_hkeyRoot, W2CT(m_sBranch), KEY_READ) != ERROR_SUCCESS )
    return ATLERROR(RegValues, L"Unable to open branch.");
  // Query how many values
  ::RegQueryInfoKey(reg,
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL,
    NULL,
    &dwCount,
    NULL,
    NULL,
    NULL,
    NULL);
  reg.Close();
  *pRetVal = dwCount;
  return S_OK;
};

STDMETHODIMP CRegValues::get__NewEnum(IUnknown** ppUnk)
{
  ATLASSERT(ppUnk);
  HRESULT Hr;
  VARIANT *arr;
  long nCount;

  *ppUnk = NULL;
  if( FAILED( get_Count(&nCount) ) ) return E_UNEXPECTED;
  if( nCount==0 ) return S_OK;

  CRegKey reg;
  USES_CONVERSION;
  if( reg.Open(m_hkeyRoot, W2CT(m_sBranch), KEY_READ) != ERROR_SUCCESS )
    return ATLERROR(RegValues, L"Unable to open branch.");

  // Now create array of VARIANTS
  arr = new VARIANT[nCount];
  if( arr==NULL ) return E_OUTOFMEMORY;

  // Create array of VARIANTs with IDispatch interfaces
  int i;
  for( i=0; i<nCount; i++ ) ::VariantInit(&arr[i]);
  for( i=0; i<nCount; i++ ) {
    // Get the item
    TCHAR szItem[MAXVALUELEN];
    DWORD cchItem = sizeof(szItem)/sizeof(TCHAR);
    DWORD dwType;
    if( ::RegEnumValue(reg, i, szItem, &cchItem, NULL, &dwType, NULL, NULL) != ERROR_SUCCESS ) break;

    // Extract value
    CComVariant vValue;
    GetValue(reg,szItem, vValue);

    // Create new COM object
    CComObject<CRegValue>* pRegValue;
    HRESULT Hr;
    Hr = CComObject<CRegValue>::CreateInstance(&pRegValue);
    if( FAILED(Hr) ) return Hr;
    InitValue(pRegValue, m_hkeyRoot, m_sBranch, szItem, vValue);

    // Ok, we need to put it into the VARIANT
    IDispatch *pDisp = NULL;
    pRegValue->QueryInterface(IID_IDispatch, (void **)&pDisp);
    V_VT(&arr[i]) = VT_DISPATCH;
    V_DISPATCH(&arr[i]) = pDisp;
  };
  reg.Close();

  // Create enumerator
  Hr = CreateEnumerator<VarArrEnum>(ppUnk, &arr[0], &arr[nCount], NULL, AtlFlagCopy);
  
  // Clean up temporary array
  for( i=0; i<nCount; i++ ) ::VariantInit(&arr[i]);
  delete [] arr;
  
  return Hr;
};

/**
 * Creates a new Registry Value.
 * <P>
 * A new <B>RegValue</B> object is returned. The new value is, however,
 * not written to the System Registry before you assign a value to the new
 * object.
 *
 * @remarks If the Registry Node already exists the method still
 *          succeeds. The previous value will be overwritten when
 *          new values are assigned to the object.
 */
STDMETHODIMP CRegValues::Add(BSTR Item, IRegValue** ppvObject)
{
  ATLASSERT(ppvObject);
  if( ppvObject==NULL ) return E_POINTER;

  CComObject<CRegValue>* pRegValue;
  HRESULT Hr;
  Hr = CComObject<CRegValue>::CreateInstance(&pRegValue);
  if( FAILED(Hr) ) return Hr;
  CComVariant vValue; // dummy
  USES_CONVERSION;
  InitValue(pRegValue, m_hkeyRoot, m_sBranch, W2CT(Item), vValue);
  return pRegValue->QueryInterface(ppvObject);
};

STDMETHODIMP CRegValues::Remove(VARIANT Item)
{
  CRegKey reg;
  TCHAR szItem[MAXVALUELEN];
  HRESULT Hr;
  USES_CONVERSION;
  // Make sure the value actually exists...
  if( FAILED( Hr = LocateValue(reg, Item, szItem, sizeof(szItem)/sizeof(TCHAR)) ) ) return Hr;
  // Open it so we can delete value
  if( reg.Open(m_hkeyRoot, W2CT(m_sBranch)) != ERROR_SUCCESS ) return E_UNEXPECTED; // Should not fail!
  if( reg.DeleteValue( szItem ) != ERROR_SUCCESS )
    return ATLERROR(RegValues, L"Unable to delete value.");
  reg.Close();
  return S_OK;
};

