// RegValue.cpp : Implementation of CRegValue

#include "stdafx.h"

#include "EasyReg.h"
#include "RegValue.h"

/**
 * <B>RegValue</B> represents a Registry value.
 * <P>
 * <B>RegValue</B> has two properties which allows you to query the <B>Name</B> of the value
 * and query and set the data contents.
 * <P>
 * The <B>Value</B> property returns the data contents of the Registry Value. The
 * result is returned in a VARIANT data type. This VARIANT's type reflects the
 * data type the value has in the System Registry.
 * Currently only 2 data types are supported:
 * <UL>
 * <LI><B>VT_BSTR</B> - string</LI>
 * <LI><B>VT_I4</B> - integer</LI>
 * </UL>
 */
/* INTERFACE IRegValue */



/////////////////////////////////////////////////////////////////////////////
// CRegValue


/////////////////////////////////////////////////////////////////////////////
// Implementation

STDMETHODIMP CRegValue::get_Name(BSTR *pVal)
{
  ATLASSERT(m_hkeyRoot!=NULL);
  ATLASSERT(pVal);
  if( pVal==NULL ) return E_POINTER; 
  return m_sName.CopyTo(pVal);
}

/**
 * Returns/sets the values of the Registry Value.
 * <P>
 * The <B>Value</B> property returns the data contents of the Registry Value. The
 * result is returned in a VARIANT data type. This VARIANT's type reflects the
 * data type the value has in the System Registry.
 * <P>
 * When you assign values to the property, the data type of the VARIANT value
 * passed as argument will
 * also be the data type written to the System Registry.
 * <P>
 * Currently only 2 data types are supported:
 * <UL>
 * <LI><B>VT_BSTR</B> - string</LI>
 * <LI><B>VT_I4</B> - integers</LI>
 * </UL>
 * Other integer data types are automatically converted to the VT_I4 type.
 */
STDMETHODIMP CRegValue::get_Value(VARIANT *pVal)
{
  ATLASSERT(m_hkeyRoot!=NULL);
  ATLASSERT(pVal);
  if( pVal==NULL ) return E_POINTER; 
  ::VariantInit(pVal);
  *pVal = m_vValue;
  return S_OK;
}

STDMETHODIMP CRegValue::put_Value(VARIANT newVal)
{
  ATLASSERT(m_hkeyRoot!=NULL);
  CRegKey reg;
  long res;
  USES_CONVERSION;
  if( reg.Create(m_hkeyRoot, W2CT(m_sBranch)) != ERROR_SUCCESS ) {
    return ATLERROR(RegValue, L"Unable to access entry.");
  };
  LPCTSTR szName;
  szName = ( m_sName.Length()==0 ? NULL : W2CT(m_sName) );
  switch( V_VT(&newVal) ) {
  case VT_BSTR:
    {
      LPCTSTR szValue = W2CT(V_BSTR(&newVal));
      if( ::lstrlen(szValue) > MAXVALUELEN ) return ATLERROR(RegValue, L"Maximum string limit reached.");
      res = reg.SetValue(szValue, szName);
    }
    break;
  case VT_I2:
    res = reg.SetValue((long)V_I2(&newVal), szName);
    break;
  case VT_I4:
    res = reg.SetValue(V_I4(&newVal), szName);
    break;
  default:
    return ATLERROR(RegValue, L"Unsupported data type.");
  };
  reg.Close();
  if( res != ERROR_SUCCESS ) return ATLERROR(RegValue, L"Unable to write data. Check permissions.");
  // Set internal value
  m_vValue = newVal;
  return S_OK;
}
