// RegNodes.cpp : Implementation of CRegNodes

#include "stdafx.h"

#include "EasyReg.h"
#include "RegNodes.h"
#include "RegNode.h"
#include "enum.h"

/**
 * <B>RegNodes</B> represents a collection of Registry Nodes.  A Registry Node is
 * also know as a System Registry Key.
 * <P>
 * <B>RegNodes</B> has methods to add a Registry Node and remove a node, and list all the attached nodes.
 * It also has the usual methods and properties for a collection: <B>Count</B>, <B>_NewEnum</B>, and <B>Item</B>.
 *
 */
/* INTERFACE IRegNodes */


/////////////////////////////////////////////////////////////////////////////
// CRegNodes


/////////////////////////////////////////////////////////////////////////////
// Implementation

HRESULT CRegNodes::LocateNode(CRegKey &reg, VARIANT Index, LPTSTR szItem/*=NULL*/, DWORD cchItem/*=0*/)
// Helper function which will initialize/open the CRegKey passed.
// The main task it to convert the VARIANT index (which can be an index position or
// a node name) to a registry key string.
// If you supply a 'szItem' buffer the new registry key string will be returned.
{
  USES_CONVERSION;
  // In case we didn't get a buffer on the argument list
  // we create one on the stack...
  if( szItem==NULL ) {
    szItem = (LPTSTR)_alloca(MAXNODELEN);
    cchItem = MAXNODELEN;
  }
  if( szItem==NULL ) return E_OUTOFMEMORY;

  // The LocateNode() function returns a complete Registry branch
  ::lstrcpy( szItem, W2CT(m_sBranch));
  ::lstrcat( szItem, _T("\\"));
  switch( V_VT(&Index) ) {
  case VT_I1:
  case VT_I2:
    ::VariantChangeType(&Index, &Index, 0, VT_I4);
    // Fall through...
  case VT_I4:
    {
      // Lookup index (position). Passed values are 1-based, but Registry index is 0-based
      if( reg.Open(m_hkeyRoot, W2CT(m_sBranch), KEY_READ) != ERROR_SUCCESS )
        return ATLERROR(RegNodes, L"Parent Node not found.");
      TCHAR szSubItem[MAXNODELEN];
      if( ::RegEnumKey(reg, V_I4(&Index)-1, szSubItem, sizeof(szSubItem)/sizeof(TCHAR)) != ERROR_SUCCESS ) 
        return ATLERROR(RegValues, L"Invalid item index.");
      ::lstrcat(szItem, szSubItem);
    };
    break;
  case VT_BSTR:
    ::lstrcat(szItem, W2CT(V_BSTR(&Index)));
    break;
  default:
    return ATLERROR(RegNodes, L"Invalid argument.");
  };
  // Ok, do a simple check to see if the entry is present.
  // This will open the CRegKey object
  if( reg.Open(m_hkeyRoot, szItem, KEY_READ) != ERROR_SUCCESS )
    return ATLERROR(RegNodes, L"Node not found.");
  return S_OK;
};

HRESULT CRegNodes::InitNode(CRegNode *pObj, HKEY hkeyRoot, LPCTSTR sBranch)
// Helper function to initialize a newly create CRegNode object
{
  ATLASSERT(hkeyRoot!=NULL);
  ATLASSERT(sBranch);
  if( pObj==NULL ) return E_POINTER;
  pObj->m_hkeyRoot = hkeyRoot;
  pObj->m_sBranch = sBranch;
  return S_OK;
};


/////////////////////////////////////////////////////////////////////////////
// IRegNodes

STDMETHODIMP CRegNodes::get_Item(VARIANT Index, IRegNode** ppvObject)
{
  if( ppvObject==NULL ) return E_POINTER;

  CRegKey reg;
  TCHAR szItem[MAXNODELEN];
  HRESULT Hr;
  if( FAILED( Hr = LocateNode(reg, Index, szItem, sizeof(szItem)/sizeof(TCHAR)) ) ) return Hr;
 
  CComObject<CRegNode>* pRegNode;
  Hr = CComObject<CRegNode>::CreateInstance(&pRegNode);
  if( FAILED(Hr) ) return Hr;
  InitNode(pRegNode, m_hkeyRoot, szItem);
 
  return pRegNode->QueryInterface(ppvObject);
};

STDMETHODIMP CRegNodes::get_Count(long* ppRetVal)
{
  ATLASSERT(ppRetVal);
  if( ppRetVal==NULL ) return E_POINTER;

  DWORD dwCount;
  CRegKey reg;
  USES_CONVERSION;
  if( reg.Open(m_hkeyRoot, W2CT(m_sBranch), KEY_READ) != ERROR_SUCCESS )
    return ATLERROR(RegNodes, L"Unable to open branch.");
  // Query how many sub-nodes
  ::RegQueryInfoKey(reg,
    NULL, 
    NULL, 
    NULL, 
    &dwCount, 
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL);
  reg.Close();
  *ppRetVal = dwCount;
  return S_OK;
};

STDMETHODIMP CRegNodes::get__NewEnum(IUnknown** ppUnk)
{
  ATLASSERT(ppUnk);
  HRESULT Hr;
  VARIANT *arr;
  long nCount;

  *ppUnk = NULL;
  if( FAILED( get_Count(&nCount) ) ) return E_UNEXPECTED;
  if( nCount==0 ) return S_OK;

  CRegKey reg;
  USES_CONVERSION;
  if( reg.Open(m_hkeyRoot, W2CT(m_sBranch), KEY_READ) != ERROR_SUCCESS )
    return ATLERROR(RegNodes, L"Unable to open branch.");

  // Now create array of VARIANTS
  arr = new VARIANT[nCount];
  if( arr==NULL ) return E_OUTOFMEMORY;

  // Create array of VARIANTs with IDispatch interfaces
  int i;
  for( i=0; i<nCount; i++ ) ::VariantInit(&arr[i]);
  for( i=0; i<nCount; i++ ) {
    // Get the item
    TCHAR szItem[MAXNODELEN];
    DWORD cchItem = sizeof(szItem)/sizeof(TCHAR);
    if( ::RegEnumKey(reg, i, szItem, cchItem) != ERROR_SUCCESS ) break;

    // Create new COM object
    CComObject<CRegNode>* pRegNode;
    HRESULT Hr;
    Hr = CComObject<CRegNode>::CreateInstance(&pRegNode);
    if( FAILED(Hr) ) return Hr;
    InitNode(pRegNode, m_hkeyRoot, szItem);

    // Ok, we need to put it into the VARIANT
    IDispatch *pDisp = NULL;
    pRegNode->QueryInterface(IID_IDispatch, (void **)&pDisp);
    V_VT(&arr[i]) = VT_DISPATCH;
    V_DISPATCH(&arr[i]) = pDisp;
  };
  reg.Close();

  // Create enumerator
  Hr = CreateEnumerator<VarArrEnum>(ppUnk, &arr[0], &arr[nCount], NULL, AtlFlagCopy);

  // Clean up temporary array
  for( i=0; i<nCount; i++ ) ::VariantClear(&arr[i]);
  delete [] arr;

  return Hr;
};

/**
 * Creates a new Registry Value.
 * <P>
 * A new <B>RegValue</B> object is returned. The new value is, however,
 * not written to the System Registry before you assign a value to the new
 * object.
 *
 * @remarks If the Registry Node already exists the method still
 *          succeeds.
 */
STDMETHODIMP CRegNodes::Add(BSTR Item, IRegNode** ppvObject)
{
  ATLASSERT(Item!=NULL);
  ATLASSERT(ppvObject);
  USES_CONVERSION;
  if( ppvObject==NULL ) return E_POINTER;
  
  LPCTSTR szItem = W2CT(Item);
  if( ::lstrlen(szItem) > MAXNODELEN ) return ATLERROR(RegNodes, L"Name exceeds maximum limit.");

  // Compose complete string
  CComBSTR sStr;
  sStr = m_sBranch;
  sStr += _T("\\");
  sStr += Item;

  // We must actually create the node now...
  // So first we open the parent branch and
  // then we create the node!
  CRegKey reg;
  HRESULT Hr;

  // Open parent key
  if( reg.Open(m_hkeyRoot, W2CT(m_sBranch), KEY_READ) != ERROR_SUCCESS )
    return ATLERROR(RegNodes, L"Unable to open branch.");
  // Create subkey
  switch( reg.Create(reg, szItem) ) {
  case ERROR_SUCCESS:
    break;
  case ERROR_ACCESS_DENIED:
    return ATLERROR(RegNode, L"Access denied.");
  default:
    return ATLERROR(RegNode, L"Unable to create Node.");
  };
  reg.Close();

  // Create RegNode object we should return to caller
  CComObject<CRegNode>* pRegNode;
  Hr = CComObject<CRegNode>::CreateInstance(&pRegNode);
  if( FAILED(Hr) ) return Hr;
  InitNode(pRegNode, m_hkeyRoot, W2CT(sStr));
  return pRegNode->QueryInterface(ppvObject);
};

STDMETHODIMP CRegNodes::Remove(BSTR Item)
{
  ATLASSERT(Item!=NULL);
  CRegKey reg;
  HRESULT Hr;
  USES_CONVERSION;
  LPCTSTR szItem = W2CT(Item);
  // Make sure the key is there...
  if( FAILED( Hr = LocateNode(reg, CComVariant(Item)) ) ) return Hr;
  // Open parent with ALL ACCESS...
  if( reg.Open(m_hkeyRoot, W2CT(m_sBranch)) != ERROR_SUCCESS ) return E_UNEXPECTED; // Should not fail!
  // On WinNT we're required to call RecurseDeleteKey() to delete
  // subitems as well.
  if( reg.RecurseDeleteKey( szItem ) != ERROR_SUCCESS )
    return ATLERROR(RegNodes, L"Unable to delete node.");
  reg.Close();
  return S_OK;
};
