// RegNode.cpp : Implementation of CRegNode

#include "stdafx.h"

#include "EasyReg.h"
#include "RegNode.h"

#include "RegNodes.h"
#include "RegValues.h"

/**
 * The <B>RegNode</B> object contains information about a Registry Node. 
 * A Registry Node is also known as a System Registry Key.
 * <P>
 * The <B>RegNode</B> object contains just a few methods and two properties:
 * <BR>
 * The <B>Name</B> property returns the complete Registry Sub key string, while
 * the <B>Root</B> property allows you to identify which Registry root the Node
 * is located in.
 * <BR>
 * <B>RegNode</B> has two methods, which return collections.  One returns
 * a collection of sub-nodes attached to the current node. The other method
 * returns a collection of the values the node contains.
 */
/* INTERFACE IRegNode */


/////////////////////////////////////////////////////////////////////////////
// CRegNode



/////////////////////////////////////////////////////////////////////////////
// Implementation



/////////////////////////////////////////////////////////////////////////////
// IRegNode

STDMETHODIMP CRegNode::get_Root(HKEY *pVal)
{
  ATLASSERT(pVal);
  ATLASSERT(m_hkeyRoot!=NULL);
  if( pVal==NULL ) return E_POINTER; 
  *pVal = m_hkeyRoot;
  return S_OK;
}

STDMETHODIMP CRegNode::get_Name(BSTR *pVal)
{
  ATLASSERT(pVal);
  ATLASSERT(m_hkeyRoot!=NULL);
  if( pVal==NULL ) return E_POINTER; 
  return m_sBranch.CopyTo(pVal);
}

/**
 * Opens a Registry Key.
 * <P>
 * Use this method to open a System Registry key.
 * The <B>RootKey</B> argument can be one of the predefined Registry Root handles:
 * <UL>
 * <LI>HKEY_CLASSES_ROOT</LI>
 * <LI>HKEY_CURRENT_USER</LI>
 * <LI>HKEY_LOCAL_MACHINE</LI>
 * <LI>HKEY_USERS</LI>
 * <LI>HKEY_PERFORMANCE_DATA</LI>
 * <LI>HKEY_CURRENT_CONFIG</LI>
 * <LI>HKEY_DYN_DATA</LI>
 * </UL>
 * <P>
 * Use the <B>ReadOnly</B> argument to specify the desired access rights you wish
 * to access the Registry key with.
 *
 * @param RootKey A valid HKEY constant.
 * @param Key The subkey to open.
 * @param ReadOnly Desired access rights.
 */
STDMETHODIMP CRegNode::Set(HKEY RootKey, BSTR Key, VARIANT_BOOL ReadOnly)
{
  ATLASSERT(RootKey!=NULL);
  ATLASSERT(Key!=NULL);
  USES_CONVERSION;
  // Set some access rights
  REGSAM Access = KEY_ALL_ACCESS;
  if( ReadOnly==VARIANT_TRUE ) Access = KEY_READ;
  // Attempt to open the key.
  // This will test if the key exists and can be accessed with
  // the requested access rights.
  // The 'ReadOnly' argument is then discarded. Access rights is
  // set individually on each access function.
  CRegKey reg;
  switch( reg.Open(RootKey, W2CT(Key), Access) ) {
  case ERROR_SUCCESS:
    break; // Ok
  case ERROR_ACCESS_DENIED:
    return ATLERROR(RegNode, L"Access denied.");
  default:
    return ATLERROR(RegNode, L"Unable to open Node.");
  };
  reg.Close();
  // Set internal values
  m_hkeyRoot = RootKey;
  m_sBranch = Key;
  return S_OK;
}

STDMETHODIMP CRegNode::get_Nodes(IRegNodes **pVal)
{
  ATLASSERT(pVal);
  ATLASSERT(m_hkeyRoot!=NULL);
  if( pVal==NULL ) return E_POINTER; 
  if( m_hkeyRoot==NULL ) return ATLERROR(RegNode, L"Not initialized.");
  CComObject<CRegNodes>* pRegNodes;
  HRESULT Hr;
  // Create collection object
  Hr = CComObject<CRegNodes>::CreateInstance(&pRegNodes);
  if( FAILED(Hr) ) return Hr;
  // Initialize collection
  pRegNodes->m_hkeyRoot = m_hkeyRoot;
  pRegNodes->m_sBranch = m_sBranch;
  // Return interface
  return pRegNodes->QueryInterface(pVal);
}

STDMETHODIMP CRegNode::get_Values(IRegValues **pVal)
{
  ATLASSERT(pVal);
  ATLASSERT(m_hkeyRoot!=NULL);
  if( pVal==NULL ) return E_POINTER; 
  if( m_hkeyRoot==NULL ) return ATLERROR(RegNode, L"Not initialized.");
  CComObject<CRegValues>* pRegValues;
  HRESULT Hr;
  // Create collection object
  Hr = CComObject<CRegValues>::CreateInstance(&pRegValues);
  if( FAILED(Hr) ) return Hr;
  // Initialize collection
  pRegValues->m_hkeyRoot = m_hkeyRoot;
  pRegValues->m_sBranch = m_sBranch;
  // Return interface
  return pRegValues->QueryInterface(pVal);
}
