// EditorView.cpp : implementation of the CEditorView class
//

#include "stdafx.h"
#include "CharAnim.h"

#include "CharAnimDoc.h"
#include "EditorView.h"
#include "MainFrm.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CEditorView

IMPLEMENT_DYNCREATE(CEditorView, CView)

BEGIN_MESSAGE_MAP(CEditorView, CView)
	//{{AFX_MSG_MAP(CEditorView)
	ON_WM_DESTROY()
	ON_WM_SIZE()
	ON_COMMAND(ID_MODE_MOVE, OnModeMove)
	ON_COMMAND(ID_MODE_ROTATE, OnModeRotate)
	ON_COMMAND(ID_MODE_SELECT, OnModeSelect)
	ON_COMMAND(ID_MODE_SIZE, OnModeSize)
	ON_COMMAND(ID_VIEW_WIREFRAME, OnViewWireframe)
	ON_COMMAND(ID_VIEW_WIREFRONT, OnViewWirefront)
	ON_COMMAND(ID_VIEW_TEXTURE, OnViewTexture)
	ON_COMMAND(ID_VIEW_SHADE, OnViewShade)
	ON_COMMAND(ID_MODE_DEFAULT, OnModeDefault)
	ON_COMMAND(ID_DIR_X, OnDirX)
	ON_COMMAND(ID_DIR_Y, OnDirY)
	ON_COMMAND(ID_DIR_Z, OnDirZ)
	ON_WM_SETFOCUS()
	ON_UPDATE_COMMAND_UI(ID_VIEW_WIREFRAME, OnUpdateViewWireframe)
	ON_UPDATE_COMMAND_UI(ID_VIEW_WIREFRONT, OnUpdateViewWirefront)
	ON_UPDATE_COMMAND_UI(ID_VIEW_TEXTURE, OnUpdateViewTexture)
	ON_UPDATE_COMMAND_UI(ID_VIEW_SHADE, OnUpdateViewShade)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
	ON_WM_CREATE()
	ON_UPDATE_COMMAND_UI(ID_DIR_X, OnUpdateDirX)
	ON_UPDATE_COMMAND_UI(ID_DIR_Y, OnUpdateDirY)
	ON_UPDATE_COMMAND_UI(ID_DIR_Z, OnUpdateDirZ)
	ON_UPDATE_COMMAND_UI(ID_MODE_DEFAULT, OnUpdateModeDefault)
	ON_UPDATE_COMMAND_UI(ID_MODE_MOVE, OnUpdateModeMove)
	ON_UPDATE_COMMAND_UI(ID_MODE_ROTATE, OnUpdateModeRotate)
	ON_UPDATE_COMMAND_UI(ID_MODE_SELECT, OnUpdateModeSelect)
	ON_UPDATE_COMMAND_UI(ID_MODE_SIZE, OnUpdateModeSize)
	ON_COMMAND(ID_EDIT_SELECTALL, OnEditSelectall)
	ON_COMMAND(ID_EDIT_REMOVEALL, OnEditRemoveall)
	ON_WM_ERASEBKGND()
	ON_COMMAND(ID_VIEW_BOTTOM, OnViewBottom)
	ON_COMMAND(ID_VIEW_RIGHT, OnViewRight)
	ON_COMMAND(ID_VIEW_TOP, OnViewTop)
	ON_COMMAND(ID_VIEW_LEFT, OnViewLeft)
	ON_UPDATE_COMMAND_UI(ID_MODEL_DELETE, OnUpdateModelDelete)
	ON_UPDATE_COMMAND_UI(ID_BONE_SHOWVERTICES, OnUpdateBoneShowvertices)
	ON_UPDATE_COMMAND_UI(ID_BONES_ASSIGN, OnUpdateBonesAssign)
	ON_UPDATE_COMMAND_UI(ID_MODEL_PROPERTIES, OnUpdateModelProperties)
	ON_WM_LBUTTONDBLCLK()
	ON_WM_RBUTTONDOWN()
	ON_COMMAND(ID_BONES_ASSIGN, OnBonesAssign)
	ON_COMMAND(ID_BONE_SHOWVERTICES, OnBoneShowvertices)
	ON_UPDATE_COMMAND_UI(ID_DIR_XY, OnUpdateDirXy)
	ON_COMMAND(ID_DIR_XY, OnDirXy)
	//}}AFX_MSG_MAP
	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_DIRECT, CView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CView::OnFilePrintPreview)
END_MESSAGE_MAP()


/////////////////////////////////////////////////////////////////////////////
// CEditorView construction/destruction
/////////////////////////////////////////////////////////////////////////////

CEditorView::CEditorView()
{
   m_bCaptured = FALSE;
   m_cfg.RenderMode = WIREFRAME;
   m_cfg.CurrentView = VIEW_VIEW;
   m_cfg.ViewDirection = DIR_CUSTOM; // will be changed in OnCreate()
   m_cfg.Direction = DIR_X;
   m_cfg.frame = 194; // HARDCODED FOR NOW
   VectorClear( m_cfg.rot );
   VectorClear( m_cfg.trans );
   VectorClear( m_cfg.camera.forward );
   VectorClear( m_cfg.camera.right );
   VectorClear( m_cfg.camera.vright );
   VectorClear( m_cfg.camera.vup );
   VectorClear( m_cfg.camera.vpn );
   m_action = TOOL_DEFAULT;
}

CEditorView::~CEditorView()
{
}

BOOL CEditorView::PreCreateWindow(CREATESTRUCT& cs)
{
	cs.lpszClass = AfxRegisterWndClass(0,
										AfxGetApp()->LoadStandardCursor(IDC_CROSS), 
										(HBRUSH)(BLACK_BRUSH),
										AfxGetApp()->LoadIcon(IDR_MAINFRAME));
   cs.style |= CS_DBLCLKS;
   m_ogl.PreCreateWindow(cs);
   return CView::PreCreateWindow(cs);
}


/////////////////////////////////////////////////////////////////////////////
// CEditorView drawing
/////////////////////////////////////////////////////////////////////////////

void CEditorView::OnDraw(CDC* pDC)
{
	CCharAnimDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
   Render();
}

BOOL CEditorView::Render()
{
   ASSERT(m_ogl.IsGLRCValid());
   if( !m_ogl.IsGLRCValid() ) return FALSE;
	CCharAnimDoc *pDoc = GetDocument();
	ASSERT_VALID(pDoc);

   // Init rendering engine
   m_ogl.InitRender(); 

   glLoadIdentity();
   glRotatef(-90,  1, 0, 0);	    // put Z going up
   glRotatef(90,  0, 0, 1);	    // put Z going up
   glRotatef(m_cfg.rot[0],  0, 1, 0);
   glRotatef(-m_cfg.rot[1],  0, 0, 1);
   glRotatef(-m_cfg.rot[2],  1, 0, 0);
   glTranslatef (-m_cfg.trans[2],  -m_cfg.trans[0],  m_cfg.trans[1]);

   BuildCameraMatrix();

   // Parse all object and render
   POSITION pos = pDoc->m_objects.GetHeadPosition();
   for( int i=0; i<pDoc->m_objects.GetCount(); i++ ) {
      CAnimObject *obj = (CAnimObject *)pDoc->m_objects.GetNext(pos);
      ASSERT_VALID(obj);
      obj->Render(m_cfg);
   };
   // Close rendering engine
   m_ogl.DoneRender();

   m_LastRenderTime = time(NULL);

   return TRUE;
}

BOOL CEditorView::DoSelect(SELECTMODE mode, CRect *rc )
{
   CCharAnimDoc* pDoc = GetDocument();
   ASSERT_VALID(pDoc);
   // Must translate rect to fit actual viewport   
   if( rc!=NULL ) m_ogl.FitRect(*rc);
   // Traverse all objects and do select on
   // all items
   POSITION pos = pDoc->m_objects.GetHeadPosition();
   for( int i=0; i<pDoc->m_objects.GetCount(); i++ ) {
      CAnimObject *obj = (CAnimObject *)pDoc->m_objects.GetNext(pos);
      ASSERT_VALID(obj);
      if( rc!=NULL ) 
         obj->Select(m_ogl.GetDC(), mode, m_cfg, *rc );
      else
         obj->Select(m_ogl.GetDC(), mode, m_cfg, CRect() );
   };
   Render();
   return TRUE;
};


/////////////////////////////////////////////////////////////////////////////
// CEditorView printing
/////////////////////////////////////////////////////////////////////////////

BOOL CEditorView::OnPreparePrinting(CPrintInfo* pInfo)
{
	// default preparation
	return DoPreparePrinting(pInfo);
}

void CEditorView::OnBeginPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: add extra initialization before printing
}

void CEditorView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: add cleanup after printing
}


/////////////////////////////////////////////////////////////////////////////
// CEditorView diagnostics
/////////////////////////////////////////////////////////////////////////////

#ifdef _DEBUG
void CEditorView::AssertValid() const
{
	CView::AssertValid();
}

void CEditorView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

CCharAnimDoc* CEditorView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CCharAnimDoc)));
	return (CCharAnimDoc*)m_pDocument;
}
#endif //_DEBUG


/////////////////////////////////////////////////////////////////////////////
// CEditorView implementation
/////////////////////////////////////////////////////////////////////////////

void CEditorView::ChangeRenderMode( RENDERMODE RenderMode )
{
   m_ogl.ChangeRenderMode(RenderMode);
   m_cfg.RenderMode = RenderMode;
	Render();
}

BOOL CEditorView::UpdateViewMatrix(TOOLMODE action, TRenderDef &cfg )
{
	int dx = m_cfg.OrgPoint.x - m_cfg.CurPoint.x;
	int dy = m_cfg.OrgPoint.y - m_cfg.CurPoint.y;

	switch(action) {
	case TOOL_MOVE:
		m_cfg.trans[0] -= dx / 5.0f;
		m_cfg.trans[1] += dy / 5.0f;
		break;
	case TOOL_ROTATE:
		m_cfg.rot[0] += (dy * 180.0f) / 500.0f;
		m_cfg.rot[1] -= (dx * 180.0f) / 500.0f;
		#define clamp(x) x = x > 360.0f ? x-360.0f : x < -360.0f ? x+=360.0f : x
		clamp(m_cfg.rot[0]);
		clamp(m_cfg.rot[1]);
		break;
	case TOOL_SIZE:
		m_cfg.trans[2] -= (dx+dy) / 4.0f;
		break;
	}
   return TRUE;
};

void CEditorView::DrawBox(CPoint &p1, CPoint &p2)
{
   if( m_action!=TOOL_SELECT ) return; // ok   
   // Initialize gdi functions
   CDC *dc = this->GetDC();
   dc->SetBkMode(TRANSPARENT);
	dc->SetROP2(R2_NOT);
	CPen penStroke;
	if (!penStroke.CreatePen(PS_DOT, 1, RGB(255,255,255)) ) return;
	CPen* pOldPen = dc->SelectObject(&penStroke);
	// Draw...
	dc->MoveTo( p1.x,p1.y );
	dc->LineTo( p1.x,p2.y );
	dc->LineTo( p2.x,p2.y );
	dc->LineTo( p2.x,p1.y );
	dc->LineTo( p1.x,p1.y );
	// Done
	dc->SelectObject(pOldPen);
   this->ReleaseDC(dc);
};

void CEditorView::UpdateViewData()
{
   CMainFrm *frm = (CMainFrm *)AfxGetMainWnd();
   ASSERT_KINDOF(CMainFrm,frm);
   //m_cfg.CurrentView = (VIEWTYPE)frm->m_CurrentViewIndex;
   // We must validate the change of view matches a valid render mode
   switch( m_cfg.CurrentView ) {
   case VIEW_VIEW:
      if( m_action==TOOL_SELECT ) m_action=TOOL_DEFAULT;
      break;
   default:
      switch( m_cfg.RenderMode ) {
      case TEXTURE:
      case SHADE:
         ChangeRenderMode(WIREFRAME);
         break;
      };
   }
};

void CEditorView::SetMessageText(LPCTSTR szText)
{
   CMDIFrameWnd *pFrame = (CMDIFrameWnd *)AfxGetMainWnd();
   if( pFrame==NULL ) return;
   pFrame->SetMessageText(szText);
}

void CEditorView::SetViewDir(VIEWDIR NewDirection)
{
#define DIST 80.0f
   if( m_cfg.ViewDirection == NewDirection ) return;
   CString s;
   switch( NewDirection ) {
   case DIR_TOP:
      s.LoadString(IDS_BOTTOM);
      VectorSet(m_cfg.trans, 0.0, -DIST, 0.0 );
      VectorSet(m_cfg.rot, -90.0, 0.0, 0.0 );
      break;
   case DIR_BOTTOM:
      s.LoadString(IDS_BOTTOM);
      VectorSet(m_cfg.trans, 0.0, DIST, 0.0 );
      VectorSet(m_cfg.rot, 90.0, 0.0, 0.0 );
      break;
   case DIR_LEFT:
      s.LoadString(IDS_LEFT);
      VectorSet(m_cfg.trans, 0.0, 0.0, -DIST );
      VectorSet(m_cfg.rot, 0.0, 0.0, 0.0 );
      break;
   case DIR_RIGHT:
      s.LoadString(IDS_RIGHT);
      VectorSet(m_cfg.trans, 0.0, 0.0, DIST );
      VectorSet(m_cfg.rot, 0.0, 180.0, 0.0 );
      break;
   case DIR_CUSTOM:
      s.LoadString(IDS_CUSTOM);
      break;
   default:
      ASSERT(FALSE);
   };
   m_cfg.ViewDirection = NewDirection;
   this->SetWindowText(s);
   Render();
}

void CEditorView::BuildCameraMatrix()
{
   float	xa, ya;
   float	matrix[4][4];
   int i;

   xa = m_cfg.rot[0]/180.0f*PI;
   ya = m_cfg.rot[1]/180.0f*PI;

   // the movement matrix is kept 2d
   m_cfg.camera.forward[0] = (float)cos(ya);
   m_cfg.camera.forward[1] = (float)sin(ya);
   m_cfg.camera.right[0] = m_cfg.camera.forward[1];
   m_cfg.camera.right[1] = -m_cfg.camera.forward[0];

   ::glGetFloatv(GL_PROJECTION_MATRIX, &matrix[0][0]);

   for (i=0 ; i<3 ; i++)
   {
      m_cfg.camera.vright[i] = matrix[i][0];
      m_cfg.camera.vup[i] = matrix[i][1];
      m_cfg.camera.vpn[i] = matrix[i][2];
   }
   VectorNormalize(m_cfg.camera.vright);
   VectorNormalize(m_cfg.camera.vup);
   VectorNormalize(m_cfg.camera.vpn);
}


/////////////////////////////////////////////////////////////////////////////
// CEditorView message handlers
/////////////////////////////////////////////////////////////////////////////

void CEditorView::OnLButtonDown(UINT nFlags, CPoint point) 
{
   if( m_action==TOOL_DEFAULT ) return;
   //
   m_cfg.OrgPoint = m_cfg.StartPoint = point;
   UpdateViewData();
   ::SetCapture(this->GetSafeHwnd());
   m_bCaptured = TRUE;
   //
   switch( m_cfg.CurrentView ) {
   case VIEW_BONES:
      {
      DoSelect(SEL_REMOVE_ALL);
      CRect rc(point,CSize(1,1));
      DoSelect(SEL_ADD,&rc);
      }
      break;
   case VIEW_CAMERA:
      break;
   default:
      if( m_action==TOOL_SELECT) DrawBox( m_cfg.StartPoint,m_cfg.OrgPoint );
      break;
   };
	
   CView::OnLButtonDown(nFlags, point);
}

void CEditorView::OnLButtonUp(UINT nFlags, CPoint point) 
{
   if( m_bCaptured ) {
      DrawBox(m_cfg.StartPoint,m_cfg.OrgPoint );
      ::ReleaseCapture();
      m_bCaptured = FALSE;
      // Ready to do action...
      switch( m_cfg.CurrentView ) {
      case VIEW_VIEW:
      case VIEW_CAMERA:
         Render();
         break;
      default:
         switch( m_action ) {
         case TOOL_SELECT:
            {
            // Do standard select action 
            CRect rc(m_cfg.StartPoint,m_cfg.OrgPoint);
            if( (nFlags & MK_SHIFT)==0 )
               DoSelect(SEL_ADD, &rc);
            else   
               DoSelect(SEL_REMOVE, &rc);
            }
            break;
         }
      };
      
   };
   
   CView::OnLButtonUp(nFlags, point);
}

void CEditorView::OnLButtonDblClk(UINT nFlags, CPoint point) 
{
   if( m_bCaptured ) {
      ::ReleaseCapture();
      m_bCaptured = FALSE;
   };
   switch( m_cfg.CurrentView ) {
   case VIEW_OBJECT:
      {
      DoSelect(SEL_REMOVE_ALL);
      CRect rc(point,CSize(1,1));
      rc.InflateRect(4,4);
      DoSelect(SEL_ADD, &rc);
      }
      break;
   case VIEW_BONES:
      if( m_cfg.NumSelections>0 ) OnBonesProperties();
      break;
   };
	CView::OnLButtonDblClk(nFlags, point);
}

void CEditorView::OnRButtonDown(UINT nFlags, CPoint point) 
{
   OnLButtonDblClk(nFlags,point);	
	//CView::OnRButtonDown(nFlags, point);
}

void CEditorView::OnMouseMove(UINT nFlags, CPoint point) 
{
   if( m_bCaptured ) {
      m_cfg.CurPoint = point;
      switch( m_cfg.CurrentView ) {
      case VIEW_VIEW:
         SetViewDir(DIR_CUSTOM);
         UpdateViewMatrix(m_action, m_cfg);
         Render();
         break;
      default:
         switch( m_action ) {
         case TOOL_SELECT:
            DrawBox( m_cfg.StartPoint, m_cfg.OrgPoint );
            DrawBox( m_cfg.StartPoint, point );
            break;
         case TOOL_MOVE:
         case TOOL_ROTATE:
         case TOOL_SIZE:
            // Do standard matrix action 
            CCharAnimDoc* pDoc = GetDocument();
	         ASSERT_VALID(pDoc);
            POSITION pos = pDoc->m_objects.GetHeadPosition();
            for( int i=0; i<pDoc->m_objects.GetCount(); i++ ) {
               CAnimObject *obj = (CAnimObject *)pDoc->m_objects.GetNext(pos);
               ASSERT_VALID(obj);
               obj->UpdateMatrix(m_action, m_cfg);
            };
            m_ogl.SetDefaults(m_cfg.RenderMode);
            Render();
            break;
         };
      };

      m_cfg.OrgPoint = point;
   };
	
	CView::OnMouseMove(nFlags, point);
}

void CEditorView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint) 
{
   UpdateViewData();
   Render();

   CView::OnUpdate(pSender,lHint,pHint);
}

void CEditorView::OnSize(UINT nType, int cx, int cy) 
{
   CView::OnSize(nType, cx, cy);

   UpdateViewData();
   m_ogl.SetDefaults(m_cfg.RenderMode);
	m_ogl.OnSize(nType, cx, cy);
   Render();
}

void CEditorView::OnDestroy() 
{
   if( m_bCaptured ) ::ReleaseCapture();
   m_bCaptured = FALSE;
	//
   CApp *app = (CApp *)AfxGetApp();
   app->m_ActiveDocument = NULL;

   CView::OnDestroy();
}

void CEditorView::OnSetFocus(CWnd* pOldWnd) 
{
	CView::OnSetFocus(pOldWnd);
   // Signal possible document change to app
   CApp *app = (CApp *)AfxGetApp();
   app->m_ActiveDocument = GetDocument();
   // Let OpenGL handle focus change too
   m_ogl.OnSetFocus(pOldWnd);
}

int CEditorView::OnCreate(LPCREATESTRUCT lpCreateStruct) 
{
   if (CView::OnCreate(lpCreateStruct) == -1)
		return -1;
   CDC *dc = new CClientDC(this);
   m_ogl.OnCreate(lpCreateStruct,dc);
   SetViewDir(DIR_LEFT);
	return 0;
}

BOOL CEditorView::OnEraseBkgnd(CDC* pDC) 
{
	return m_ogl.OnEraseBkgnd(pDC);
   //return CView::OnEraseBkgnd(pDC);
}

/////////////////////////////////////////////////////////////////////////////

void CEditorView::OnModeDefault() 
{
   m_action = TOOL_DEFAULT;
}

void CEditorView::OnModeMove() 
{
   m_action = TOOL_MOVE;
   m_cfg.Direction=DIR_XY;
}

void CEditorView::OnModeRotate() 
{
   m_action = TOOL_ROTATE;
   if( m_cfg.Direction==DIR_XY ) m_cfg.Direction=DIR_X;
}

void CEditorView::OnModeSelect() 
{
   m_action = TOOL_SELECT;
}

void CEditorView::OnModeSize() 
{
   m_action = TOOL_SIZE;
}

void CEditorView::OnViewWireframe() 
{
   ChangeRenderMode(WIREFRAME);
}

void CEditorView::OnViewWirefront() 
{
   ChangeRenderMode(WIREFRONT);
}

void CEditorView::OnViewShade() 
{
   ChangeRenderMode(SHADE);
}

void CEditorView::OnViewTexture() 
{
   ChangeRenderMode(TEXTURE);
}

void CEditorView::OnDirX() 
{
   m_cfg.Direction = DIR_X;
}

void CEditorView::OnDirY() 
{
   m_cfg.Direction = DIR_Y;
}

void CEditorView::OnDirZ() 
{
   m_cfg.Direction = DIR_Z;
}

void CEditorView::OnDirXy() 
{
   m_cfg.Direction = DIR_XY;
}

void CEditorView::OnEditSelectall() 
{
   DoSelect(SEL_ADD_ALL);
}

void CEditorView::OnEditRemoveall() 
{
   DoSelect(SEL_REMOVE_ALL); 
}

void CEditorView::OnViewBottom() 
{
   SetViewDir(DIR_BOTTOM);
}

void CEditorView::OnViewRight() 
{
   SetViewDir(DIR_RIGHT);
}

void CEditorView::OnViewTop() 
{
   SetViewDir(DIR_TOP);
}

void CEditorView::OnViewLeft() 
{
   SetViewDir(DIR_LEFT);
}

void CEditorView::OnBonesAssign() 
{
	CCharAnimDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
   CSelectDlg dlg;
   CAnimFigure *obj = pDoc->GetSelectedFigure();
   if( obj==NULL ) return;
   obj->m_bones.GetBones(dlg.m_members, &obj->m_bones.m_Skeleton);
   dlg.DoModal();
   if( dlg.m_result.GetSize()!=1 ) return;
   obj->AssignVertextToBone(dlg.m_result[0]);
   Render();
}

void CEditorView::OnBoneShowvertices() 
{
	CCharAnimDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
   CSelectDlg dlg;
   CAnimFigure *obj = pDoc->GetSelectedFigure();
   if( obj==NULL ) return;
   obj->m_bones.GetBones(dlg.m_members, &obj->m_bones.m_Skeleton);
   dlg.DoModal();
   if( dlg.m_result.GetSize()!=1 ) return;
   DoSelect(SEL_REMOVE_ALL);
   obj->SelectAssigned(dlg.m_result[0]);
   Render();
}

void CEditorView::OnBonesProperties()
{
	CCharAnimDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
   CAnimFigure *obj = pDoc->GetSelectedFigure();
   if( obj==NULL ) return;
   tBone *pBone = obj->m_bones.GetSelectedBone();
   if( pBone==NULL ) return;
   CPositionDlg dlg;
   dlg.m_RotX = (int)pBone->rot[0];
   dlg.m_RotY = (int)pBone->rot[1];
   dlg.m_RotZ = (int)pBone->rot[2];
   if( dlg.DoModal()==IDOK ) {
      pBone->rot[0] = (float)dlg.m_RotX;
      pBone->rot[1] = (float)dlg.m_RotY;
      pBone->rot[2] = (float)dlg.m_RotZ;
      Render();
   };
};


/////////////////////////////////////////////////////////////////////////////
// CEditorView update handlers
/////////////////////////////////////////////////////////////////////////////

void CEditorView::OnUpdateViewWireframe(CCmdUI* pCmdUI) 
{
   pCmdUI->SetCheck(m_cfg.RenderMode==WIREFRAME);
}

void CEditorView::OnUpdateViewWirefront(CCmdUI* pCmdUI) 
{
   pCmdUI->SetCheck(m_cfg.RenderMode==WIREFRONT);
}

void CEditorView::OnUpdateViewTexture(CCmdUI* pCmdUI) 
{
   pCmdUI->SetCheck(m_cfg.RenderMode==TEXTURE);
   pCmdUI->Enable(m_cfg.CurrentView==VIEW_VIEW);
}

void CEditorView::OnUpdateViewShade(CCmdUI* pCmdUI) 
{
   pCmdUI->SetCheck(m_cfg.RenderMode==SHADE);
   pCmdUI->Enable(m_cfg.CurrentView==VIEW_VIEW);
}

void CEditorView::OnUpdateModeDefault(CCmdUI* pCmdUI) 
{
   pCmdUI->SetCheck( m_action==TOOL_DEFAULT );
}

void CEditorView::OnUpdateModeMove(CCmdUI* pCmdUI) 
{
   pCmdUI->SetCheck( m_action==TOOL_MOVE );
}

void CEditorView::OnUpdateModeRotate(CCmdUI* pCmdUI) 
{
   pCmdUI->SetCheck( m_action==TOOL_ROTATE );
}

void CEditorView::OnUpdateModeSelect(CCmdUI* pCmdUI) 
{
   BOOL hidden=FALSE;
   hidden |= (m_cfg.CurrentView==VIEW_VIEW);
   hidden |= (m_cfg.CurrentView==VIEW_BONES);
   pCmdUI->Enable( !hidden );
   pCmdUI->SetCheck( m_action==TOOL_SELECT );
}

void CEditorView::OnUpdateModeSize(CCmdUI* pCmdUI) 
{
   BOOL hidden=FALSE;
   hidden |= (m_cfg.CurrentView==VIEW_BONES);
   pCmdUI->Enable( !hidden );
   pCmdUI->SetCheck( m_action==TOOL_SIZE );
}

void CEditorView::OnUpdateDirX(CCmdUI* pCmdUI) 
{
   BOOL hidden=FALSE;
   hidden |= (m_cfg.CurrentView==VIEW_VIEW);
   hidden |= (m_action!=TOOL_ROTATE) && (m_action!=TOOL_MOVE);
   pCmdUI->Enable( !hidden );
   pCmdUI->SetCheck( m_cfg.Direction==DIR_X );
}

void CEditorView::OnUpdateDirY(CCmdUI* pCmdUI) 
{
   BOOL hidden=FALSE;
   hidden |= (m_cfg.CurrentView==VIEW_VIEW);
   hidden |= (m_action!=TOOL_ROTATE) && (m_action!=TOOL_MOVE);
   pCmdUI->Enable( !hidden );
   pCmdUI->SetCheck( m_cfg.Direction==DIR_Y );
}

void CEditorView::OnUpdateDirZ(CCmdUI* pCmdUI) 
{
   BOOL hidden=FALSE;
   hidden |= (m_cfg.CurrentView==VIEW_VIEW);
   hidden |= (m_action!=TOOL_ROTATE); // only show on ROTATE
   pCmdUI->Enable( !hidden );
   pCmdUI->SetCheck( m_cfg.Direction==DIR_Z );
}

void CEditorView::OnUpdateDirXy(CCmdUI* pCmdUI) 
{
   BOOL hidden=FALSE;
   hidden |= (m_cfg.CurrentView==VIEW_VIEW);
   hidden |= (m_action!=TOOL_MOVE); // only show on "MOVE"
   pCmdUI->Enable( !hidden );
   pCmdUI->SetCheck( m_cfg.Direction==DIR_XY );
}

void CEditorView::OnUpdateBoneShowvertices(CCmdUI* pCmdUI) 
{
   pCmdUI->Enable( m_cfg.NumSelections==1 && m_cfg.CurrentView==VIEW_VERTEX );
}

void CEditorView::OnUpdateBonesAssign(CCmdUI* pCmdUI) 
{
   pCmdUI->Enable( m_cfg.NumSelections==1 && m_cfg.CurrentView==VIEW_VERTEX );
}

void CEditorView::OnUpdateModelProperties(CCmdUI* pCmdUI) 
{
   pCmdUI->Enable( m_cfg.NumSelections==1 );
}

void CEditorView::OnUpdateModelDelete(CCmdUI* pCmdUI) 
{
   pCmdUI->Enable( m_cfg.NumSelections>0 );
}

