// AnimBones.cpp: implementation of the CAnimBones class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "CharAnim.h"
#include "AnimBones.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CAnimBones::CAnimBones()
{
   m_Skeleton.children = NULL;
   m_Skeleton.CV_weight = NULL;
   m_SelectedBoneId = -1;
   m_Count = 0;
}

CAnimBones::~CAnimBones()
{
   Destroy();
}

BOOL CAnimBones::LoadBones(LPCTSTR szFilename, int Vertices)
{
   CStdioFile f;
   CString s;
   m_nModelVertices = Vertices;
   TRY {

      CFileException e;
      if (f.Open( szFilename, CFile::modeRead|CFile::shareExclusive, &e ) != TRUE ) 
         return FALSE;

      if( f.ReadString(s)==FALSE) return FALSE;
      if( s!=_T("BONES") ) {
         AfxMessageBox(IDS_ERR_BADBONEFILE);
         return FALSE; // too many bones
      };

      // Reset current skeleton
      Destroy();

      CString cmd;
      int count = 1;
      int i=-1; // will be incremented to 0 right away
                // since s==_T("#")
      float x,y,z;
      tBone *p = NULL;
      tBone *bones = NULL;
      while( TRUE ) {
         // Read attribute
         if( f.ReadString(s)==FALSE) break;
         s.TrimLeft();
         if( s.IsEmpty() ) continue;
         if( s[0]==_T(';') ) continue;
         if( s[0]==_T('#') ) {
            // Get pointer to bone structure
            if( i>=0 ) 
               p = &bones[i];
            else
               p = &m_Skeleton;
            ASSERT(p);
            if( p==NULL ) return FALSE; // error: something is definitily wrong
            // Set default attributes
            p->parent = NULL;
            p->children = NULL;
            p->brothers = NULL;
            p->CV_weight = new float[m_nModelVertices];
            // Initially, assign all vertices to Bones object
            for( int j=0; j<m_nModelVertices; j++ ) 
               p->CV_weight[j] = ( i==0 ? 1.0f : 0.0f );
            VectorClear( p->rot );
            VectorClear( p->rot_min );
            VectorClear( p->rot_max );
            VectorClear( p->trans );
            VectorSet( p->scale, 1.0, 1.0, 1.0 );
            MatrixClear( p->matrix );
            // Get ready for next bone (when file reads another #-char)
            i++;
            if( i > count ) {
               AfxMessageBox(IDS_ERR_TOOFEWBONES);
               return FALSE; // too many bones
            };
            // Read another string from file please
            continue;
         };
         ASSERT(p);
         if( p==NULL ) {
            AfxMessageBox(IDS_ERR_BADBONEFILE);
            return FALSE; // too many bones
         };
         s.TrimRight();
         cmd = BfxRemoveToken(s,_T(' '));
         cmd.MakeUpper();
         s.TrimLeft();
         // Set attribute value
         if( cmd==_T("ID") ) 
            p->Id = atol(s);
         else if( cmd==_T("NAME") ) 
            p->Name = s;
         else if( cmd==_T("NUMBER") ) {
            // This attribute is only permitted in the
            // beginning of the .bon file and defines
            // how many bones in total there is
            int j;
            count = atol(s);
            bones = new tBone[count];
            for( j=0; j<count; j++ ) {
               bones[j].Id = 0;
               bones[j].children = NULL;
               bones[j].brothers = NULL;
            };
            m_Count = count;
         }
         if( cmd==_T("ROT") ) {
            sscanf(s, _T("%f %f %f"), &x,&y,&z );
            VectorSet( p->rot, x,y,z );
         }
         if( cmd==_T("ROT_MIN") ) {
            sscanf(s, _T("%f %f %f"), &x,&y,&z );
            VectorSet( p->rot_min, x,y,z );
         }
         if( cmd==_T("ROT_MAX") ) {
            sscanf(s, _T("%f %f %f"), &x,&y,&z );
            VectorSet( p->rot_max, x,y,z );
         }
         else if( cmd==_T("TRANS") ) {
            sscanf(s, _T("%f %f %f"), &x,&y,&z );
            VectorSet( p->trans, x,y,z );
         }
         else if( cmd==_T("SCALE") ) {
            sscanf(s, _T("%f %f %f"), &x,&y,&z );
            VectorSet( p->scale, x,y,z );
         }         
         else if( cmd==_T("PARENT") ) {
            int id = atol(s);
            tBone *parent = FindBone(id,&m_Skeleton);
            if( parent!=NULL ) {
               p->parent = parent;
               p->brothers = parent->children;
               parent->children = p;
            }
            else {
               ASSERT(FALSE);
               AfxMessageBox(IDS_ERR_BADBONE);
            };
         }
         else if( cmd==_T("ASSIGN") ) {
            while( !s.IsEmpty() ) {
               CString tok( BfxRemoveToken(s,_T(' ')) );
               int id = atol(tok);
               ASSERT(id>=0);
               ASSERT(id<m_nModelVertices);
               if( id<0 || id>=m_nModelVertices ) break;
               ASSERT(p);
               ASSERT(p->CV_weight);
               p->CV_weight[id] = 1.0f;
               bones[0].CV_weight[id] = 0.0f; // clear default weight
            };
         }
      };

      f.Close();
   }
   CATCH_ALL(e) {
		// Error
      f.Abort();
      return FALSE;
	}
	END_CATCH_ALL;

   //
   // Prepare bones
   //
   // We need to get each bone matrix initialized
   // with the viewport values
   GetBonesMatrix();
   
   return TRUE;
};

BOOL CAnimBones::Destroy()
{
   int i;
   // This will delete the complete array of bones
   // since it is allocated as one big block
   if( m_Skeleton.CV_weight!=NULL ) {
      delete [] m_Skeleton.CV_weight;
      m_Skeleton.CV_weight = NULL;
   };
   tBone *pBone = m_Skeleton.children;
   if( pBone!=NULL ) {
      for( i=0; i<m_Count; i++ ) delete [] pBone[i].CV_weight;
      delete [] m_Skeleton.children;
      m_Skeleton.children = NULL;
   };
   //
   m_SelectedBoneId = -1;
   return TRUE;
};


//////////////////////////////////////////////////////////////////////
// Implementation
//////////////////////////////////////////////////////////////////////

BOOL CAnimBones::Render( TRenderDef &cfg )
{
   // Do render
   glPushMatrix();

   // Set root skeleton's orientation and position
   glTranslatef(m_Skeleton.trans[0], 
                m_Skeleton.trans[1], 
                m_Skeleton.trans[2]);
   glRotatef(m_Skeleton.rot[2], 0.0f, 0.0f, 1.0f);
   glRotatef(m_Skeleton.rot[1], 0.0f, 1.0f, 0.0f);
   glRotatef(m_Skeleton.rot[0], 1.0f, 0.0f, 0.0f); 

   BOOL ret = Render(cfg,&m_Skeleton);

   glPopMatrix();
   
   // Get bones matrix
   GetBonesMatrix();

   // Done
   return ret;
};

BOOL CAnimBones::Render( TRenderDef &cfg, tBone *pBone )
{
   if(pBone==NULL) return TRUE;

   glPushMatrix();

   // Set base orientation and position
   glTranslatef(pBone->trans[0], pBone->trans[1], pBone->trans[2]);

   glRotatef(pBone->rot[2], 0.0f, 0.0f, 1.0f);
   glRotatef(pBone->rot[1], 0.0f, 1.0f, 0.0f);
   glRotatef(pBone->rot[0], 1.0f, 0.0f, 0.0f); 

   //
   // Draw arrow
   //
   if( cfg.CurrentView!=VIEW_SELECT ) {
      glPushMatrix();
      glScalef(4.0,4.0,4.0);
      glBegin(GL_LINES);
      glColor3f(1.0f, 0.0f, 0.0f);	// X Axis starts - color red
      glVertex3f(-0.2f,  0.0f, 0.0f);
      glVertex3f( 0.2f,  0.0f, 0.0f);
      glVertex3f( 0.2f,  0.0f, 0.0f);	// Top piece of arrowhead
      glVertex3f( 0.15f,  0.04f, 0.0f);
      glVertex3f( 0.2f,  0.0f, 0.0f);	// Bottom piece of arrowhead
      glVertex3f( 0.15f, -0.04f, 0.0f);
      glColor3f(0.0f, 1.0f, 0.0f);	// Y Axis starts - color green
      glVertex3f( 0.0f,  0.2f, 0.0f);
      glVertex3f( 0.0f, -0.2f, 0.0f);			
      glVertex3f( 0.0f,  0.2f, 0.0f);	// Top piece of arrowhead
      glVertex3f( 0.04f,  0.15f, 0.0f);
      glVertex3f( 0.0f,  0.2f, 0.0f);	// Bottom piece of arrowhead
      glVertex3f( -0.04f,  0.15f, 0.0f);
      glColor3f(0.0f, 0.0f, 1.0f);	// Z Axis starts - color blue
      glVertex3f( 0.0f,  0.0f,  0.2f);
      glVertex3f( 0.0f,  0.0f, -0.2f);
      glVertex3f( 0.0f,  0.0f, 0.2f);	// Top piece of arrowhead
      glVertex3f( 0.0f,  0.04f, 0.15f);
      glVertex3f( 0.0f, 0.0f, 0.2f);	// Bottom piece of arrowhead
      glVertex3f( 0.0f, -0.04f, 0.15f);
      glEnd();
      glPopMatrix();
   };

   // The scale is local so we push and pop
   glPushMatrix();
   glScalef(pBone->scale[0], pBone->scale[1], pBone->scale[2]);

   //
   // Draw bone structure
   //
   if(pBone->children!=NULL) {
      // Choose a color
      if( m_SelectedBoneId==pBone->Id )
         glColor3f(1.0f, 1.0f, 1.0f);	// Selected bone is dull Yellow
      else
         glColor3f(0.4f, 0.4f, 0.0f);	// UnSelected bone is dull Yellow

      // In case of a select feedback, we need to determine the
      // ID of the bone, not the vertex
      if( cfg.CurrentView==VIEW_SELECT ) {
         glPassThrough((GLfloat)pBone->Id);
         glBegin(GL_POINTS);
      }
      else
         glBegin(GL_LINE_STRIP);
      glVertex3f( 0.0f,  0.4f, 0.0f);		// 0
      glVertex3f(-0.4f,  0.0f,-0.4f);		// 1
      glVertex3f( 0.4f,  0.0f,-0.4f);		// 2
      glVertex3f( 0.0f,  pBone->children->trans[1], 0.0f); // Base
      glVertex3f(-0.4f,  0.0f,-0.4f);		// 1
      glVertex3f(-0.4f,  0.0f, 0.4f);		// 4
      glVertex3f( 0.0f,  0.4f, 0.0f);		// 0
      glVertex3f( 0.4f,  0.0f,-0.4f);		// 2
      glVertex3f( 0.4f,  0.0f, 0.4f);		// 3
      glVertex3f( 0.0f,  0.4f, 0.0f);		// 0
      glVertex3f(-0.4f,  0.0f, 0.4f);		// 4
      glVertex3f( 0.0f,  pBone->children->trans[1], 0.0f); // Base
      glVertex3f( 0.4f,  0.0f, 0.4f);		// 3
      glVertex3f(-0.4f,  0.0f, 0.4f);		// 4
      glEnd();
   };

   glPopMatrix();	// This pop is just for the scale

   // Recursive call the remaining bones
   Render(cfg,pBone->children);

	glPopMatrix();	// This pop for the whole

   // Recursive call this bone's brothers
   Render(cfg,pBone->brothers);

   return TRUE;
};


BOOL CAnimBones::UpdateMatrix(TOOLMODE action, TRenderDef &cfg )
{
	int dx = cfg.OrgPoint.x - cfg.CurPoint.x;
	int dy = cfg.OrgPoint.y - cfg.CurPoint.y;

   if( m_SelectedBoneId==-1 ) return FALSE;
   tBone *pBone = FindBone(m_SelectedBoneId,&m_Skeleton);
   if( pBone==NULL ) return FALSE;

	switch(action) {
	case TOOL_MOVE:
		pBone->trans[0] -= dx / 5.0f;
		pBone->trans[1] += dy / 5.0f;
		break;
	case TOOL_ROTATE:
		pBone->rot[0] += (dy * 180.0f) / 500.0f;
		pBone->rot[1] -= (dx * 180.0f) / 500.0f;
		#define clamp(x) x = x > 360.0f ? x-360.0f : x < -360.0f ? x+=360.0f : x
		clamp(pBone->rot[0]);
		clamp(pBone->rot[1]);
		break;
	case TOOL_ZOOM:
		pBone->trans[2] -= (dx+dy) / 4.0f;
		break;
	}
   return TRUE;
};

//////////////////////////////////////////////////////////////////////
// Operations
//////////////////////////////////////////////////////////////////////

tBone * CAnimBones::FindBone(int Id, tBone *pBone)
// Find a bone based on the Id
{
   if( pBone==NULL ) return NULL;
   if( pBone->Id==Id ) return pBone;
   tBone *p;
   if( (p = FindBone( Id, pBone->brothers ))!=NULL ) return p;
   if( (p = FindBone( Id, pBone->children ))!=NULL ) return p;
   return NULL;
}

tBone * CAnimBones::FindBone(CString &Name, tBone *pBone)
// Find a bone based on the Name
{
   if( pBone==NULL ) return NULL;
   if( pBone->Name.CompareNoCase(Name)==0 ) return pBone;
   tBone *p;
   if( (p = FindBone( Name, pBone->brothers ))!=NULL ) return p;
   if( (p = FindBone( Name, pBone->children ))!=NULL ) return p;
   return NULL;
}

tBone *CAnimBones::GetSelectedBone()
// Returns the currently selected bone
{
   if( m_SelectedBoneId<0 ) return NULL; // nothing selected
   return FindBone(m_SelectedBoneId,&m_Skeleton);
};

BOOL CAnimBones::GetBonesMatrix()
// A helper function that initializes each bone matrix.
// This is in a seperate function since we need the
// MODEVIEW matrix without the eye/camera projection
// so we have to initialize the matrix first before
// traversing the bones individually and assigning each
// bone matrix 
{
   glPushMatrix();   

   glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective (40, 1, 1,800);

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

   //glTranslatef(0.0F, 0.0F, -70.0F);
	glRotatef(-90.0F, 1.0F, 0.0F, 0.0F); // make z up
	glRotatef(90.0F, 0.0F, 0.0F, 1.0F);
   
   // Set root skeleton's orientation and position
   glTranslatef(m_Skeleton.trans[0], 
                m_Skeleton.trans[1], 
                m_Skeleton.trans[2]);
   glRotatef(m_Skeleton.rot[2], 0.0f, 0.0f, 1.0f);
   glRotatef(m_Skeleton.rot[1], 0.0f, 1.0f, 0.0f);
   glRotatef(m_Skeleton.rot[0], 1.0f, 0.0f, 0.0f); 

   BOOL ret = GetBonesMatrix(&m_Skeleton);

   glPopMatrix();
   return ret;
};

BOOL CAnimBones::GetBonesMatrix(tBone *pBone)
// Recursively tranvers all bones and
// assign the view matrix to the in-memory
// buffer
{
   if(pBone==NULL) return TRUE;

   glPushMatrix();

   // Set base orientation and position
   glTranslatef(pBone->trans[0], pBone->trans[1], pBone->trans[2]);

   glRotatef(pBone->rot[2], 0.0f, 0.0f, 1.0f);
   glRotatef(pBone->rot[1], 0.0f, 1.0f, 0.0f);
   glRotatef(pBone->rot[0], 1.0f, 0.0f, 0.0f); 

   // The scale is local so we push and pop
   glPushMatrix();
   glScalef(pBone->scale[0], pBone->scale[1], pBone->scale[2]);

   // Grab the matrix at this point so we can use it for the deformation
   glGetFloatv(GL_MODELVIEW_MATRIX, pBone->matrix);

   glPopMatrix();	// This pop is just for the scale

   // Recursive call the remaining bones
   GetBonesMatrix(pBone->children);

	glPopMatrix();	// This pop for the whole

   GetBonesMatrix(pBone->brothers);

   return TRUE;
};

BOOL CAnimBones::GetBones(CStringArray &output, tBone *pBone)
{
   if( pBone==NULL ) return TRUE;
   output.Add(pBone->Name);
   GetBones(output,pBone->children);
   GetBones(output,pBone->brothers);
   return TRUE;
};

BOOL CAnimBones::GetBones(tBone *output, tBone *pBone)
{
   return FALSE;
};
