/////////////////////////////////////////////////////////////////////////////
// idlhelp.cpp : The IDL / CPP scanner
//
// Written by Bjarke Viksoe (bjarke@viksoe.dk)
// Copyright (c) 2001 Bjarke Viksoe.
//
// This is the source code for the utility that scans
// IDL and CPP files and outputs HTML documentation.
// It is the result of 3 hours of MFC hacking.
// Beware! It is poorly written, not at all extensible in
// a pleasing way, no lexical parsing, C style coding 
// and certainly contains a bug or two.
//
// This code may be used in compiled form in any way you desire. This
// file may be redistributed by any means PROVIDING it is 
// not sold for profit without the authors written consent, and 
// providing that this notice and the authors name is included. 
//
// This file is provided "as is" with no expressed or implied warranty.
// The author accepts no liability if it causes any damage to you or your
// computer whatsoever. It's free, so don't hassle me about it.
//
// Beware of bugs.
//

#include "stdafx.h"
#include "idlhelp.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define ADDFULLSTOP(x) ( x.Right(1)==_T(".") ? x : x + _T(".") )

CWinApp theApp;

using namespace std;


typedef struct {
   CString AttribString;
   long Id;
   CString sId;
   BOOL IsHidden;
   BOOL IsProperty;
   CString HelpString;
} tAttributes;

typedef struct {
   CString Name;
   tAttributes Attrib;
   CString DataType;
} tArgument;

typedef struct {
   CString Name;
   CString HelpString;
} tReturnValue;

typedef struct {
   CString Name;
   CString ReturnType;
   BOOL    HasPointers;
   BOOL    HasOptionals;
   BOOL    HasVariants;
   BOOL    HasReturnValue;
   BOOL    IsProperty;
   int     nArgs;
   tArgument Args[20];
} tFunction;

typedef struct {
   CString Name;
   CString Description;
   //
   long nArgs;
   tArgument Args[20];
   //
   long nReturns;
   tReturnValue Returns[20];
   //
   CString Version;
   CString Author;
   CString Remarks;
   CString Sample;
   //
   BOOL IsHidden;
} tSourceDescription;

typedef struct {
   CString Name;
   CString Default;
   CStringArray FilterNames;
   CStringArray FilterValues;
} tDefaultText;

struct tagSettings {
   CStringArray SearchPaths;
   CString sFileExtension;
   CString sTextHeader;
   CString sTextFooter;
   CString sStyleSheet;
   CString sTOCFilename;
   CString sIgnoreLinks;
   // Default argument descriptions
   int nDefaults;
   tDefaultText Defaults[30];
   // Default error code strings
   struct tagErrorStrings {
      CString sS_OK;
      CString sE_POINTER;
      CString sE_INVALIDARG;
      CString sE_UNEXPECTED;
      CString sE_FAIL;
   } ErrorStrings;
   // Default html tags
   struct tagTags {
      CString sBodyBegin;
      CString sBodyEnd;
      CString sCodeBegin;
      CString sCodeEnd;
      CString sTitleBegin;
      CString sTitleEnd;
   } Tags;
} Config;

CIni ini;


/////////////////////////////////////////////////////////////////////////////
// 

CStringArray SeeAlso;

void AddSeeAlso(CStringArray &SeeAlso, LPCTSTR Title, LPCTSTR Filename)
{
   CString s;
   s = Title;
   // Don't include standard interfaces (IUnknown, IDispatch etc)
   if( Config.sIgnoreLinks.Find(s + _T(",")) >= 0 ) return;
   // Construct SeeAlso entry
   s += _T("|");
   // Make sure we don't add duplicates
   for( int i=0; i<SeeAlso.GetSize(); i++ )
      if( SeeAlso[i].Find(s)==0 ) return; // already added
   s += Filename;
   // Add string to array
   SeeAlso.Add(s);
};


/////////////////////////////////////////////////////////////////////////////
// 

CString HtmlFormat(LPCTSTR Text)
{
   if( Text==NULL ) return CString(); // can very well be NULL
   if( !AfxIsValidString(Text) ) return CString(); // memory error
   CString str;
   int len = _tcslen( Text );
   LPTSTR d = str.GetBuffer( len*10*sizeof(TCHAR) ); //make some room!
   ASSERT( d );
   if( d==NULL ) return CString();
   LPCTSTR s = Text;
   for( int i=0; i<len; i++ ) {
      switch( *s ) {
      case _T('&'): // &
         *d = _T('&'); d = _tcsinc(d);
         *d = _T('a'); d = _tcsinc(d);
         *d = _T('m'); d = _tcsinc(d);
         *d = _T('p'); d = _tcsinc(d);
         *d = _T(';'); d = _tcsinc(d);
         break;
      case _T('\t'): // TAB
         {
            for( int j=0; j<5; j++ ) {
               *d = _T('&'); d = _tcsinc(d);
               *d = _T('n'); d = _tcsinc(d);
               *d = _T('b'); d = _tcsinc(d);
               *d = _T('s'); d = _tcsinc(d);
               *d = _T('p'); d = _tcsinc(d);
               *d = _T(';'); d = _tcsinc(d);
            };
         };
         break;
      case _T('<'): // <
         *d = _T('&'); d = _tcsinc(d);
         *d = _T('l'); d = _tcsinc(d);
         *d = _T('t'); d = _tcsinc(d);
         *d = _T(';'); d = _tcsinc(d);
         break;
      case _T('>'): // >
         *d = _T('&'); d = _tcsinc(d);
         *d = _T('g'); d = _tcsinc(d);
         *d = _T('t'); d = _tcsinc(d);
         *d = _T(';'); d = _tcsinc(d);
         break;
      case _T('\r'): // CR
         // skip (browsers don't like these)
         break;
      case _T('\n'): // EOL
         *d = _T('<'); d = _tcsinc(d);
         *d = _T('B'); d = _tcsinc(d);
         *d = _T('R'); d = _tcsinc(d);
         *d = _T('>'); d = _tcsinc(d);
         // fall through
      default:
         *d=*s; // copy valid char
         d = _tcsinc(d);
      };
      s = _tcsinc(s);
   };
   *d=_T('\0'); // make sure it is null-terminated
   str.ReleaseBuffer();
   return str;
};

BOOL FileReadString(CStdioFile &f, CString &s)
// Wrapper around the CStdioFile::ReadString function.
// We need to format the string.
{
   BOOL ret = f.ReadString(s);
   s.Replace(_T('\t'),_T(' '));
   s.TrimLeft();
   s.TrimRight();
   return ret;
};

CString GetWord(CString &s)
// Parses the string for the next word (alphabethic chars only).
{
   CString ret;
   s.TrimLeft();
   while( TRUE ) {
      if( s.IsEmpty() ) break;
      if( !_istalpha(s[0]) ) break;
      ret += s[0];
      s = s.Mid(1);
   };
   s.TrimRight();
   return ret;
};


/////////////////////////////////////////////////////////////////////////////
// 

CString sTOC_Constants;
CString sTOC_Structs;
CString sTOC_Interfaces;
int nTOC_Constants;
int nTOC_Structs;
int nTOC_Interfaces;

void CreateTOCStart(CString &TOC, LPCTSTR Name, LPCTSTR SubPath)
{
   ASSERT(AfxIsValidString(Name));
   ASSERT(AfxIsValidString(SubPath));
   TOC.Format(_T("\t\t<LI> <OBJECT type=\"text/sitemap\">\n")
                _T("\t\t\t<param name=\"Name\" value=\"%s\">\n")
                _T("\t\t\t<param name=\"Local\" value=\"%sall.htm\">\n")
                _T("\t\t</OBJECT>\n")
              _T("\t\t<UL>\n"),
                (LPCTSTR)Name,
                (LPCTSTR)SubPath);
};

void CreateTOCEnd(CString &TOC)
{
   TOC += _T("\t\t</UL>\n");
};

void AddTOC(CString &TOC, int &nCount, LPCTSTR Name, LPCTSTR SubPath, LPCTSTR Title)
{
   ASSERT(AfxIsValidString(Name));
   ASSERT(AfxIsValidString(SubPath));
   CString s;
   s.Format(_T("\t\t\t<LI> <OBJECT type=\"text/sitemap\">\n")
     _T("\t\t\t\t<param name=\"Name\" value=\"%s\">\n")
     _T("\t\t\t\t<param name=\"Local\" value=\"%s%s.htm\">\n")
     _T("\t\t\t</OBJECT>\n"),
       (LPCTSTR)Title,
       (LPCTSTR)SubPath,
       (LPCTSTR)Name);
   TOC += s;
   nCount++;
};

void InitTOC(LPCTSTR SubPath)
{
   ASSERT(AfxIsValidString(SubPath));
   CString sPath;
   sPath = CString(SubPath) + _T("constants\\");
   CreateTOCStart(sTOC_Constants, _T("Enumerated Types"), sPath);
   sPath = CString(SubPath) + _T("structs\\");
   CreateTOCStart(sTOC_Structs, _T("Structures"), sPath);
   sPath = CString(SubPath) + _T("interfaces\\");
   CreateTOCStart(sTOC_Interfaces, _T("Interfaces"), sPath);
   nTOC_Constants = 0;
   nTOC_Structs = 0;
   nTOC_Interfaces = 0;
};

void DoneTOC()
{
   CreateTOCEnd(sTOC_Constants);
   CreateTOCEnd(sTOC_Structs);
   CreateTOCEnd(sTOC_Interfaces);
};

void CreateTOC(LPCTSTR Path)
{
   ASSERT(AfxIsValidString(Path));
   CString sTOC;
   if( nTOC_Constants>0 ) sTOC += sTOC_Constants;
   if( nTOC_Structs>0 ) sTOC += sTOC_Structs;
   if( nTOC_Interfaces>0 ) sTOC += sTOC_Interfaces;

   CStdioFile f_out;
   CString sFilename;
   sFilename = CString(Path) + Config.sTOCFilename;
   f_out.Open(sFilename,CFile::modeWrite|CFile::typeText|CFile::modeCreate);
   f_out.WriteString(sTOC);
   f_out.Close();
};


/////////////////////////////////////////////////////////////////////////////
// 

CStringArray SourceFile;

void ParseCodeTextString(CString &Text, CString &str)
{
   if( !Text.IsEmpty() ) {
      if( str.Right(1)==_T(">") )
         Text += _T("\n");
      else
         Text += _T(" ");
   }
   Text += str;
};

void ParseCodeSeeAlso(LPCTSTR Text, LPCTSTR Classname, CString &Title, CString &URL)
// Translates a java documentation @seealso tag into an URL.
// The 'ClassName' argument is the current class we parsing.
// The function returns the display title (text) and the resulting
// URL.
{
   ASSERT(AfxIsValidString(Text));
   ASSERT(AfxIsValidString(Classname));
   CString s(Text);
   if( s.IsEmpty() ) return;
   if( s[0]==_T('#') ) {
      // It's a link to a method in this class
      Title.Format(_T("%s::%s"), Classname, (LPCTSTR)s.Mid(1));
      URL.Format(_T("%s.htm"), (LPCTSTR)s.Mid(1));
   }
   else {
      CString Class;
      Class = BfxRemoveToken(s,_T('#'));
      if( s.IsEmpty() ) {
         // It's a link to a class
         Title.Format(_T("%s"), (LPCTSTR)Class);
         URL.Format(_T("..\\..\\%s\\%s.htm"), (LPCTSTR)Class, (LPCTSTR)Class);
      }
      else {
         // It's a link to a mehtod in a different class
         Title.Format(_T("%s::%s"), (LPCTSTR)Class,(LPCTSTR)s);
         URL.Format(_T("..\\..\\%s\\methods\\%s.htm"), (LPCTSTR)Class, (LPCTSTR)s);
      };
   };
};

CString LocateFile(LPCTSTR Filename)
// Searches for a file.
// If the file does not have a full path, we need to search
// each entry in the the added SearchPath list.
{
   ASSERT(AfxIsValidString(Filename));
   CString sFilename(Filename);
   CString sCompleteFilename;
   // The source file never has the "I"-interface prefix
   if( (sFilename[0]==_T('I')) && (sFilename[1]==_totupper(sFilename[1])) )
      sFilename = sFilename.Mid(1);
   BOOL bFound = FALSE;
   for( int i=0; i<Config.SearchPaths.GetSize(); i++ ) {
      sCompleteFilename = Config.SearchPaths[i] + sFilename + Config.sFileExtension;
      if( BfxFileExists(sCompleteFilename) ) { bFound=TRUE; break; }
   };
   if( !bFound ) return CString();
   return Config.SearchPaths[i] + sFilename;
};

void LoadSourceFile(LPCTSTR Filename)
// Cache the file
{
   ASSERT(AfxIsValidString(Filename));
   static CString sLastSourceFile;
   CString sFile = LocateFile(Filename);
   if( sFile.IsEmpty() ) return;

   CStdioFile f;
   CString sFilename;
   CString s;

   sFilename = sFile + Config.sFileExtension;
   if( sLastSourceFile == sFilename ) return; // already loaded

   SourceFile.RemoveAll();

   // Load source file (if any)
   if( BfxFileExists(sFilename) ) {
      f.Open(sFilename,CFile::typeText|CFile::modeRead);
      while( f.ReadString(s)==TRUE ) {
         // Skip the most obvious CODE lines.
         // We're really only interested in lines which looks
         // like a /*...*/ comment.
         if( _tcsncmp(s,_T("   "),3)==0 ) continue;
         if( _tcsncmp(s,_T("\t"),1)==0 ) continue;
         if( _tcsncmp(s,_T("//"),2)==0 ) continue;
         s.Replace(_T('\t'),_T(' '));
         s.TrimLeft();
         s.TrimRight();
         if( s.IsEmpty() ) continue;
         // Add good lines to source database
         SourceFile.Add(s);
      }
      f.Close();
   };
   sLastSourceFile = sFilename;
};

BOOL ParseSourceCodeDocumentation(CString &sClassName, 
                                  CString &sFunctionName, 
                                  tSourceDescription &Info)
// Look for the method or property 'sFunctionName' in the
// class 'sClassName'.
// It will load the cpp file with the 'sClassName' name.
// Returns the function description in 'Info'.
{
   int i;

   // Reset values
   Info.Name.Empty();
   Info.Description.Empty();
   Info.nArgs = 0;
   Info.nReturns = 0;
   Info.IsHidden = FALSE;

   LoadSourceFile(sClassName);
   if( SourceFile.GetSize()==0 ) return FALSE;

   // Need to locate the actual implementation of
   // the function
   CString sSearchStr1;
   CString sSearchStr2;
   CString sSearchStr3;
   CString sSearchStr4;
   sSearchStr1.Format(_T("STDMETHODIMP C%s::%s("), (LPCTSTR)sClassName.Mid(1), (LPCTSTR)sFunctionName );
   sSearchStr2.Format(_T("STDMETHODIMP C%s::get_%s("), (LPCTSTR)sClassName.Mid(1), (LPCTSTR)sFunctionName );
   sSearchStr3.Format(_T("STDMETHODIMP C%s::put_%s("), (LPCTSTR)sClassName.Mid(1), (LPCTSTR)sFunctionName );
   sSearchStr4.Format(_T("INTERFACE %s"), (LPCTSTR)sClassName );
   for( i=0; i<SourceFile.GetSize(); i++ ) {
      // Scan the entire source code file for lines which might be
      // what we're looking for...
      if( (sFunctionName.IsEmpty()) && (SourceFile[i].Find(sSearchStr4)>=0) ) break;
      // The remaining lines need the STDMETHODIMP keyword
      if( SourceFile[i].Find(_T("STDMETHODIMP"))!=0 ) continue;
      if( SourceFile[i].Find(sSearchStr1)==0 ) break;
      if( SourceFile[i].Find(sSearchStr2)==0 ) break;
      if( SourceFile[i].Find(sSearchStr3)==0 ) break;
   };
   if( i>=SourceFile.GetSize() ) return FALSE; // not found
   // or then we need to find the start of the
   // code description
   i--;
   while( i>=0 ) {
      if( SourceFile[i].IsEmpty() ) return FALSE;
      if( SourceFile[i].Left(3)==_T("/**") ) break;
      if( SourceFile[i][0]!=_T('*') ) return FALSE; // end of multi-line comment
      i--;
   };
   // If we get here, this would be the
   // top of the description field (the "/**" line to
   // be exact).
   i++;

   enum { UNKNOWN, DESCRIPTION, PARAM, RETURNVALUE, VERSION, SEEALSO, AUTHOR, REMARKS, SAMPLE } state;
   state = DESCRIPTION; // start off in "Description" mode
   Info.Name = sFunctionName;

   CString line = SourceFile[i];
   while( line.Left(1)==_T("*") ) {
      line = line.Mid(1); // remove "*"
      line.TrimLeft();
      if( line==_T("/") ) break; // ends with "*/" of which only the "/" is left here
      if( line.IsEmpty() ) goto ParseNext;
      //
      // Parse possible info token
      //
      if( line[0]==_T('@') ) {
         int pos = line.Find(_T(' '));
         if( pos==0 ) goto ParseNext; // error
         CString cmd = line.Mid(1,pos-1);
         line = line.Mid(pos);
         line.TrimLeft();
         cmd.MakeLower();
         if( cmd==_T("param") ) {
            Info.nArgs++;
            Info.Args[Info.nArgs-1].Name = BfxRemoveToken(line,_T(' '));
            state = PARAM;
         }
         else if( cmd==_T("version") ) {
            state = VERSION;
         }
         else if( cmd==_T("author") ) {
            state = AUTHOR;
         }
         else if( cmd==_T("remarks") ) {
            state = REMARKS;
         }
         else if( cmd==_T("sample") ) {
            state = SAMPLE;
         }
         else if( cmd==_T("see") ) {
            state = SEEALSO;
         }
         else if( cmd==_T("return") ) {
            Info.nReturns++;
            Info.Returns[Info.nReturns-1].Name = BfxRemoveToken(line,_T(' '));
            state = RETURNVALUE;
         }
         else if( cmd==_T("hidden") ) {
            Info.IsHidden = TRUE;
            state = UNKNOWN;
         }
         else
            state = UNKNOWN; // don't care about the remaining tokens
      };
      // Format string
      line.TrimLeft();
      //
      // Now, parse states
      //
      switch( state ) {
      case UNKNOWN:
         break;
      case PARAM:
         ParseCodeTextString( Info.Args[Info.nArgs-1].Attrib.HelpString, line );
         break;
      case RETURNVALUE:
         ParseCodeTextString( Info.Returns[Info.nReturns-1].HelpString, line );
         break;
      case AUTHOR:
         ParseCodeTextString( Info.Author, line );
         break;
      case REMARKS:
         ParseCodeTextString( Info.Remarks, line );
         break;
      case SAMPLE:
         ParseCodeTextString( Info.Sample, line );
         break;
      case SEEALSO:
         {
            CString Title;
            CString URL;
            ParseCodeSeeAlso(line,sClassName,Title,URL);
            AddSeeAlso(SeeAlso,Title,URL);
         }
         break;
      case VERSION:         
         ParseCodeTextString( Info.Version, line);
         break;
      case DESCRIPTION:
         ParseCodeTextString( Info.Description, line );
         break;
      };
ParseNext:
      // Parse next
      i++;
      line = SourceFile[i];
   };
   return TRUE;
};


/////////////////////////////////////////////////////////////////////////////
// 

CStringArray Keywords;

void HtmlInit(CStdioFile &f, LPCTSTR Title, int depth=2)
// Write HTML page header
{
   ASSERT(AfxIsValidString(Title));
   CString s;
   f.WriteString(_T("<html>\n"));
   f.WriteString(_T("<head>\n"));
   // Add META tags
   f.WriteString(_T("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=windows-1252\">\n"));   
   f.WriteString(_T("<meta name=\"GENERATOR\" content=\"BV\">\n"));   
   // Add possible STYLESHEET
   if( !Config.sStyleSheet.IsEmpty() ) {
      CString sDepth;
      for( int i=0; i<depth; i++ ) sDepth += _T("../");
      s.Format(_T("<link rel=\"stylesheet\" type=\"text/css\" href=\"%s%s\">\n"), 
         (LPCTSTR)sDepth, (LPCSTR)Config.sStyleSheet);
      f.WriteString(s);
   };   
   // Write HTML document title
   s.Format(_T("<title>%s</title>\n"), Title );
   f.WriteString(s);
   // Done with HEAD
   f.WriteString(_T("</head>\n"));   

   // Prepare BODY
   f.WriteString(Config.Tags.sBodyBegin);
   // Write default anchor
   f.WriteString(Config.Tags.sTitleBegin);
   s.Format(_T("<a name=\"%s\"></a>%s"), Title, Title);
   f.WriteString(s);
   f.WriteString(Config.Tags.sTitleEnd);
   // Write document header if available
   if( !Config.sTextHeader.IsEmpty() ) {
      f.WriteString(Config.sTextHeader);
      f.WriteString(_T("\n"));
   };   
   // Init state variables
   Keywords.RemoveAll();
   SeeAlso.RemoveAll();
};

void HtmlKeywords(CStdioFile &f)
// Write HTML keywork links
{
   if( Keywords.GetSize()>0 ) {
      f.WriteString(_T("\n<object type=\"application/x-oleobject\" classid=\"clsid:1e2a7bd0-dab9-11d0-b93a-00c04fc99f9e\">\n"));
      for( int i=0; i<Keywords.GetSize(); i++ ) {
         CString sKeyword(Keywords[i]);
         if( sKeyword.IsEmpty() ) continue;
         CString s;
         switch( sKeyword[0] ) {
         case _T('*'):
            s.Format(_T("  <param name=\"ALink Name\" value=\"%s\">\n"), (LPCTSTR)sKeyword.Mid(1));
            break;
         default:
            s.Format(_T("  <param name=\"Keyword\" value=\"%s\">\n"), (LPCTSTR)sKeyword);
            break;
         };
         f.WriteString(s);
      };
      f.WriteString(_T("</object>\n"));
   };
   Keywords.RemoveAll();
};

void HtmlDone(CStdioFile &f)
// Write HTML page footer
{
   if( !Config.sTextFooter.IsEmpty() ) {
      f.WriteString(Config.sTextFooter);
      f.WriteString(_T("\n"));
   };
   HtmlKeywords(f);
   f.WriteString(_T("<p>&nbsp;</p>\n"));
   f.WriteString(Config.Tags.sBodyEnd);
   f.WriteString(_T("</html>\n"));
};


/////////////////////////////////////////////////////////////////////////////
// 

void WriteFuncDef(CStdioFile &f, 
                  CString sFunc,
                  tFunction &fdef)
// Writes out the "method" or "property" defintion.
{
   CString tmp = sFunc;
   tmp.Format(_T("<b>%s %s"), (LPCTSTR)fdef.ReturnType, (LPCTSTR)fdef.Name);
   f.WriteString(tmp);
   if( fdef.nArgs==0 ) {
      f.WriteString(_T("()</b>"));
   }
   else {
      f.WriteString(_T("(\n"));
      for( int i=0; i<fdef.nArgs; i++ ) {
         tmp.Format(_T("  %s</b><i> %s</i>%s<b>\n"), 
            (LPCTSTR)fdef.Args[i].DataType,
            (LPCTSTR)fdef.Args[i].Name,
            i<fdef.nArgs-1 ? _T(",") : _T(""));
         f.WriteString(tmp);
      };
      f.WriteString(_T(");</b>\n"));
   };
};

void WriteProps(CStdioFile &f,
                tFunction &fdef,
                tSourceDescription &info)
// Writes out the arguments of a "property" or "method" defintion
// To be used in the "Members" section of the html page.
{
   CString tmp;
   for( int i=0; i<fdef.nArgs; i++ ) {
      // Figure out the description
      CString ArgName( fdef.Args[i].Name );
      CString sDescription;
      if( !fdef.Args[i].Attrib.HelpString.IsEmpty() )
         sDescription = fdef.Args[i].Attrib.HelpString;
      else {
         for( int j=0; j<info.nArgs; j++ ) {
            if( info.Args[j].Name.CompareNoCase(ArgName)==0 )
               sDescription = info.Args[j].Attrib.HelpString;
         };
      }
      if( sDescription.IsEmpty() ) {
         // In case we still haven't got a description (lazy programmer)
         // we might think one up...
         for( int i=0; i<Config.nDefaults; i++ ) {
            if( Config.Defaults[i].Name.CompareNoCase(ArgName)==0 ) {
               sDescription = Config.Defaults[i].Default;
               for( int j=0; j<Config.Defaults[i].FilterNames.GetSize(); j++ ) {
                  if( Config.Defaults[i].FilterNames[j].CompareNoCase( fdef.Name )==0 )
                     sDescription = Config.Defaults[i].FilterValues[j];
               };
            };
         };
      };

      tmp.Format(_T("<dt><i>%s</i></dt>\n"), (LPCTSTR)ArgName);
      f.WriteString(tmp);
      tmp.Format(_T("<dd>%s %s</dd>\n"), 
         (LPCTSTR)fdef.Args[i].Attrib.AttribString,
         (LPCTSTR)ADDFULLSTOP(sDescription));
      f.WriteString(tmp);
   };
};



/////////////////////////////////////////////////////////////////////////////
// 

void ParseAttributes(CStdioFile &f, CString &s, tAttributes &attribs)
{
   s.TrimLeft();
   if( s.Left(1)!=_T("[") ) return; // does not have attributes

   // Extract the "[attrib1, attrib2...]" string for later use
   int pos1 = s.Find(_T("["));
   int pos2 = s.Find(_T("]"), pos1>0 ? pos1 : 0);
   attribs.AttribString.Empty();
   if( (pos1>=0) && (pos2>=0) ) attribs.AttribString = s.Mid(pos1,(pos2-pos1)+1);

   // Reset
   attribs.Id = 0;
   attribs.sId.Empty();
   attribs.HelpString.Empty();
   attribs.IsHidden=FALSE;
   attribs.IsProperty=FALSE;
   
   // Parse
   s = s.Mid(1); // remove "["
   CString tmp;
   while( TRUE ) {
      s.TrimLeft();
      if( s.IsEmpty() ) break;
      if( s[0]==_T(']') ) break;
      if( s[0]==_T(',') ) s = s.Mid(1); // skip to next argument
      s.TrimLeft();

      // get attribute
      CString ret = GetWord(s);
      if( ret.IsEmpty() ) break;
      ret.MakeLower();

      if( ret==_T("id") ) {
         BfxRemoveToken(s,_T("("));
         tmp = BfxRemoveToken(s,_T(")"));
         attribs.Id = _ttol(tmp);
         attribs.sId = tmp;
      }
      else if( ret==_T("helpstring") ) {
         BfxRemoveToken(s,_T("("));
         BfxRemoveToken(s,_T("\""));
         // Parse "helpstring"
         int pos = s.Find(_T("\""));
         attribs.HelpString = s.Left(pos);
         attribs.HelpString.Replace(_T("\\r"),_T(""));
         attribs.HelpString.Replace(_T("\\n"),_T("\n"));
         attribs.HelpString.Replace(_T("\\t"),_T("\t"));
         attribs.HelpString = HtmlFormat(attribs.HelpString);
         // Get past the "helpstring" attribute
         s = s.Mid(pos);
         BfxRemoveToken(s,_T(")"));
      }
      else if( ret==_T("hidden") ) {
         attribs.IsHidden = TRUE;
      }
      else if( ret==_T("restricted") ) {
         attribs.IsHidden = TRUE;
      }
      else if( ret==_T("propget") ) {
         attribs.IsProperty = TRUE;
      }
      else if( ret==_T("propput") ) {
         attribs.IsProperty = TRUE;
      }
      else if( ret==_T("propputref") ) {
         attribs.IsProperty = TRUE;
      };

      // Find the end or next attrib...
      int pos = s.FindOneOf(_T(",]"));
      if( pos>=0 ) s = s.Mid(pos);

      if( s.IsEmpty() ) FileReadString(f,s);
   };

   // Done
   s = s.Mid(1); // remove "]"
   s.TrimLeft();
   if( s.IsEmpty() ) FileReadString(f,s);
};

void ParseArgument(CStdioFile &f, CString &s, tArgument &arg)
{
   if( s.IsEmpty() ) return;

   if( s[0]==_T('[') ) {
      ParseAttributes( f, s, arg.Attrib );
      s.TrimLeft();
   };

   CString token;
   int pos = s.FindOneOf(_T(",)"));
   if( pos<0 ) return; // error: malformed argument

   // Look for types with embedded () (e.g. SAFEARRAY(VARIANT)...)
   int pos_pfix = s.FindOneOf(_T("("));
   if( pos_pfix>=0 && pos_pfix<pos ) {
      pos = s.Find(_T(')'), pos_pfix+1);
      if( pos<0 ) return; // error: malformed argument with embedded parenthis
      // FindOneOf(",)") again...
      CString sCopy = s;
      pos = sCopy.Find(_T('(')); 
      sCopy.SetAt(pos, _T(' '));
      pos = sCopy.Find(_T(')')); 
      if( pos<0 ) return; // no matching )-char
      sCopy.SetAt(pos, _T(' '));
      //...
      pos = sCopy.FindOneOf(_T(",)"));
      if( pos<0 ) return; // error: malformed argument
   }

   token = s.Left(pos);
   s = s.Mid(pos);
   pos = token.FindOneOf(_T(" *"));
   if( pos>0 ) {
      arg.Name = token.Mid(pos);
      arg.DataType = token.Left(pos);
   }
   arg.DataType.TrimLeft();
   arg.Name.TrimLeft();
   arg.DataType.TrimRight();
   arg.Name.TrimRight();

   // In case of "*"-char at first position of name
   // E,g "IUnit *pVal" is parsed as:
   // DataType: IUnit
   // Name: *pVal
   // and this is now right! We must correct this.
   if( arg.Name.GetLength()>0 ) {
      while( arg.Name[0]==_T('*') ) {
         arg.DataType += _T(" *");
         arg.Name = arg.Name.Mid(1);
         arg.Name.TrimLeft();
      };
   }
};

void ParseFunctionDef(CStdioFile &f, LPCTSTR FuncDef, tFunction &func)
{
   CString s(FuncDef);
   // Reset
   func.nArgs = 0;
   func.HasOptionals = FALSE;
   func.HasPointers = FALSE;
   func.Name.Empty();
   func.ReturnType.Empty();
   
   // Parse simple stuff
   func.ReturnType = BfxRemoveToken(s,_T(" ")); // is very likely to be "HRESULT"
   func.Name = BfxRemoveToken(s,_T("("));
   func.HasVariants = (s.Find(_T("VARIANT"))>=0 );
   func.HasReturnValue = (s.Find(_T("retval"))>=0 );

   // Parse arguments
   while( TRUE ) {
      s.TrimLeft();
      s.TrimRight();
      if( s.IsEmpty() ) break;
      if( s[0]==_T(')') ) break;
      if( s[0]==_T(',') ) s = s.Mid(1); // skip argument      
      s.TrimLeft();
      if( s.IsEmpty() ) {
         FileReadString(f,s);
         continue;         
      };
      
      ParseArgument(f, s, func.Args[func.nArgs]);
      
      // Some settings
      func.HasPointers = (func.Args[func.nArgs].DataType.Find(_T('*'))>=0 );
      func.HasOptionals = (func.Args[func.nArgs].Attrib.AttribString.Find(_T("optional"))>=0 );

      // Ok, next argument
      func.nArgs++;
   };
   // Done
   BfxRemoveToken(s,_T(")"));
};


/////////////////////////////////////////////////////////////////////////////
// 

void ParseFunc(CStdioFile &f, 
               LPCTSTR szPath,
               CString &sClassName,
               CString &sName,
               tAttributes &attribs,
               CString &sFunc)
// Parse IDL "method" or "property" defintions
{
   if( sName.IsEmpty() ) return;
   if( sFunc.IsEmpty() ) return;

   CString sLowerName(sName);
   CString sLowerClassName(sClassName);
   CString tmp;
   int i;
   sLowerName.MakeLower();
   sLowerClassName.MakeLower();

   CDir dir;
   CString sPath;
   sPath.Format(_T("%sinterfaces\\%s\\methods"), szPath, (LPCTSTR)sLowerClassName );
   dir.Create(sPath);
   ADDBACKSLASH(sPath);

   CString sFilename;
   sFilename.Format(_T("%s%s.htm"),
      (LPCTSTR)sPath,
      (LPCTSTR)sLowerName);

   // Create new file
   CStdioFile f_out;
   f_out.Open(sFilename,CFile::modeWrite|CFile::typeText|CFile::modeCreate);
   CString sTitle;
   sTitle.Format(_T("%s::%s"), (LPCTSTR)sClassName, (LPCTSTR)sName );
   HtmlInit(f_out,sTitle,4);

   // Add self to keywords
   Keywords.Add(sName);
   Keywords.Add(sClassName + _T(" class members"));
   // Add self to SeeAlso links
   tmp.Format(_T("../%s.htm"), 
      (LPCTSTR)sClassName);
   AddSeeAlso(SeeAlso,sClassName,tmp);


   // Get info from source file is possible
   tSourceDescription SourceInfo;
   ParseSourceCodeDocumentation(sClassName,sName,SourceInfo);

   //
   // Function description
   //
   if( !SourceInfo.Description.IsEmpty() ) {
      tmp.Format(_T("<p>%s</p>\n"), (LPCTSTR)ADDFULLSTOP(SourceInfo.Description));
      f_out.WriteString(tmp);
   }
   else if( !attribs.HelpString.IsEmpty() ) {
      tmp.Format(_T("<p>%s</p>\n"), (LPCTSTR)ADDFULLSTOP(attribs.HelpString));
      f_out.WriteString(tmp);
   };

   //
   // Function header
   //
   f_out.WriteString(Config.Tags.sCodeBegin);
   tFunction fdef;
   ParseFunctionDef(f, sFunc, fdef);
   WriteFuncDef(f_out, sFunc, fdef);

   // This is pretty crap-code, but we need to determine
   // if this property has a "propput" function too, that we wish to
   // add
   tFunction fdef_prop;
   BOOL bHasTwin = FALSE;
   if( attribs.IsProperty ) {
      CString sID;
      DWORD filepos = f.GetPosition();
      FileReadString(f,tmp);
      sID.Format(_T("id(%s)"), (LPCTSTR)attribs.sId );
      if( tmp.Find(sID) >= 0 ) {
         // Found one
         bHasTwin = TRUE;
         tAttributes attribs;
         ParseAttributes(f, tmp, attribs);
         ParseFunctionDef(f, tmp, fdef_prop);
         f_out.WriteString(_T("<br>\n"));
         WriteFuncDef(f_out, tmp, fdef_prop);
      }
      else {
         f.Seek(filepos,CFile::begin);
      }     
   };

   f_out.WriteString(Config.Tags.sCodeEnd);

   //
   // Parameters
   //
   if( fdef.nArgs>0 ) {
      f_out.WriteString(_T("\n<h4>Parameters</h4>\n"));
      f_out.WriteString(_T("<dl>\n"));
      WriteProps(f_out,fdef,SourceInfo);
      if( bHasTwin && (fdef_prop.nArgs>0) ) {
         WriteProps(f_out,fdef_prop,SourceInfo);
      };
      f_out.WriteString(_T("</dl>\n"));
   };
   
   //
   // Return value
   //
#define WRITEERRORSTRING(x) \
   f_out.WriteString(_T("<dd>")); \
   f_out.WriteString(x); \
   f_out.WriteString(_T("</dd>\n")); \

   f_out.WriteString(_T("\n<h4>Return Values</h4>\n"));
   f_out.WriteString(_T("<dl>\n"));
   // Add standard error codes
   f_out.WriteString(_T("<dt>S_OK</dt>\n"));
   WRITEERRORSTRING(Config.ErrorStrings.sS_OK);
   if( fdef.HasPointers ) {
      f_out.WriteString(_T("<dt>E_POINTER</dt>\n"));
      WRITEERRORSTRING(Config.ErrorStrings.sE_POINTER);
   };
   BOOL bAddArgErr = FALSE;
   for( i=0; i<fdef.nArgs; i++ ) {
      if( fdef.Args[i].Name==_T("DomainID")) bAddArgErr = TRUE;
      if( fdef.Args[i].Name==_T("ID")) bAddArgErr = TRUE;
      if( fdef.Args[i].Name==_T("Type")) bAddArgErr = TRUE;
   };
   if( bAddArgErr ) {
      f_out.WriteString(_T("<dt>E_INVALIDARG</dt>\n"));
      WRITEERRORSTRING(Config.ErrorStrings.sE_INVALIDARG);
   };
   BOOL bAddUnexpectedErr = TRUE;
   if( attribs.IsProperty && (fdef.nArgs==1) ) bAddUnexpectedErr = FALSE;
   if( bAddUnexpectedErr ) {
      f_out.WriteString(_T("<dt>E_UNEXPECTED</dt>\n"));
      WRITEERRORSTRING(Config.ErrorStrings.sE_UNEXPECTED);
   };
   if( !attribs.IsProperty ) {
      f_out.WriteString(_T("<dt>E_FAIL</dt>\n"));
      WRITEERRORSTRING(Config.ErrorStrings.sE_FAIL);
   };
   // Now write out any additional added Return values
   for( i=0; i<SourceInfo.nReturns; i++ ) {
      CString s;
      s.Format(_T("<dt>%s</dt>\n"), (LPCTSTR)SourceInfo.Returns[i].Name);
      f_out.WriteString(s);
      s.Format(_T("<dd>%s</dd>\n"), (LPCTSTR)ADDFULLSTOP(SourceInfo.Returns[i].HelpString));
      f_out.WriteString(s);
   };
   f_out.WriteString(_T("</dl>\n"));

   //
   // Remarks
   //
   if( !SourceInfo.Remarks.IsEmpty() ) {
      f_out.WriteString(_T("\n<h4>Remarks</h4>\n"));
      tmp.Format(_T("<p>%s</p>\n"), (LPCTSTR)ADDFULLSTOP(SourceInfo.Remarks));
      f_out.WriteString(tmp);
   }

   //
   // Sample
   //
   if( !SourceInfo.Sample.IsEmpty() ) {
      f_out.WriteString(_T("\n<h4>Sample</h4>\n"));
      tmp.Format(_T("<p>%s</p>\n"), (LPCTSTR)ADDFULLSTOP(SourceInfo.Sample));
      f_out.WriteString(tmp);
   }

   //
   // See also
   //
   // Scan for Interface arguments we need to add
   for( i=0; i<fdef.nArgs; i++ ) {
      tmp = fdef.Args[i].DataType;
      if( tmp.GetLength() < 3 ) continue;
      if( (tmp[0]==_T('I')) && 
          (tmp[1]==_totupper(tmp[1])) &&
          (tmp[2]==_totlower(tmp[2])) ) { // Find 'I' as in the I in e.g 'IUnits'
         CString sArgClassName = GetWord(tmp); // strip any "*" and trailing stuff
         tmp.Format(_T("../../%s/%s.htm"), 
            (LPCTSTR)sArgClassName, 
            (LPCTSTR)sArgClassName);
         AddSeeAlso(SeeAlso,sArgClassName,tmp);
      };
   };
   // Now parse the entries and add to html
   if( SeeAlso.GetSize() > 0 ) {
      f_out.WriteString(_T("\n<h4>See Also</h4>\n"));
      f_out.WriteString(_T("<p>\n"));
      for( int i=0; i<SeeAlso.GetSize(); i++ ) {
         CString sTitle, sFilename;
         sFilename = SeeAlso[i];
         sTitle = BfxRemoveToken(sFilename,_T('|'));
         sFilename.MakeLower();
         sFilename.Replace(_T("\\"),_T("/"));
         tmp.Format(_T("<a href=\"%s\"><b>%s</b></a>%s"), 
            (LPCTSTR)sFilename, 
            (LPCTSTR)sTitle,
            ( i==SeeAlso.GetUpperBound() ? _T("") : _T(", ") ) );
         f_out.WriteString(tmp);
      };
      f_out.WriteString(_T("</p>\n"));
   };

   // Done
   HtmlDone(f_out);
   f_out.Close();
   // pheew...
};

void ParseEnum(CStdioFile &f, 
               LPCTSTR szPath,
               LPCTSTR szSubPath,
               CStdioFile &flist,
               CString &s)
//
// Parse IDL "enum" defintions
//
{
   if( s.Find(_T("enum"))<0 ) return; // not an enum

   BfxRemoveToken(s,_T(" ")); // remove "typedef"
   tAttributes attribs;
   ParseAttributes(f, s,attribs);

   if( BfxRemoveToken(s,_T(" "))!=_T("enum") ) return; // error
   CString sName;
   sName = BfxRemoveToken(s,_T(" "));

   CString sFilename;
   CString sLowerName(sName);
   sLowerName.MakeLower();
   sFilename.Format(_T("%sconstants\\%s.htm"), szPath, (LPCTSTR)sLowerName);

   CString sPath;
   sPath.Format(_T("%sconstants\\"), szSubPath);
   AddTOC(sTOC_Constants, nTOC_Constants, sLowerName, sPath, sName);

   // Add entry to overview html file
   CString tmp;
   tmp.Format(_T("<p class=indent1><a href=\"%s.htm\"><b>%s</b></a></p>\n"),
      (LPCTSTR)sLowerName,
      (LPCTSTR)sName);
   flist.WriteString(tmp);

   // Create the new entry html file
   CStdioFile f_out;
   f_out.Open(sFilename,CFile::modeWrite|CFile::typeText|CFile::modeCreate);
   HtmlInit(f_out,sName);
   Keywords.Add(_T("Enumerated Types"));
   Keywords.Add(sName);

   if( !attribs.HelpString.IsEmpty() ) {
      tmp.Format(_T("<p>%s</p>\n"), attribs.HelpString);
      f_out.WriteString(tmp);
   };
   f_out.WriteString(Config.Tags.sCodeBegin);
   tmp.Format(_T("<b>typedef enum %s {</b>\n"), (LPCTSTR)sName);
   f_out.WriteString(tmp);

   BOOL bHasHelp = FALSE;
   CString sMembers;

   while( FileReadString(f,s)==TRUE ) {
      // Process enun
      // Outer loop:
      s.TrimLeft();
      if( s.IsEmpty() ) continue;
      if( s.Left(2)==_T("//") ) continue;
      if( s.Left(1)==_T("}") ) break; // end of enum

      // Parse
      if( s.Left(1)==_T("[") ) {
         tAttributes attribs;
         ParseAttributes(f, s,attribs);
         //
         s.TrimLeft();
         if( s.IsEmpty() ) FileReadString(f,s);
         s.TrimLeft();
         //         
         if( !attribs.HelpString.IsEmpty() ) {
            // Attribute has an "helpstring" property, which we will
            // add later in the "Members:" html section.
            int pos = s.FindOneOf(_T(" ,=["));
            if( pos>=0 ) tmp = s.Left(pos); else tmp = s;
            sMembers += _T("<tr valign=top>\n");
            sMembers += _T("<td width=45%%><b>") + tmp + _T("</b></td>\n");
            sMembers += _T("<td width=55%%>") + attribs.HelpString + _T("</td>\n");
            sMembers += _T("</tr>\n");
            bHasHelp = TRUE;
         };
      };
      int pos = s.Find(_T("//")); if( pos>=0 ) s = s.Left(pos); // remove "//" comment
      tmp.Format(_T("    %s\n"), (LPCTSTR)s);
      f_out.WriteString(tmp);
   };

   tmp.Format(_T("<b>} %s</b>"), (LPCTSTR)sName);
   f_out.WriteString(tmp);
   f_out.WriteString(Config.Tags.sCodeEnd);

   f_out.WriteString(_T("\n<h4>Members</h4>"));
   if( !bHasHelp ) {
      // no HELPSTRING for individual members
      f_out.WriteString(_T("<p>See individual functions.</p>"));
   }
   else {
      f_out.WriteString(_T("<table cellspacing=4 cols=2>\n"));
      f_out.WriteString(_T("<tr valign=top>\n"));
      f_out.WriteString(_T("<th align=left width=45%%>Value</th>\n"));
      f_out.WriteString(_T("<th align=left width=55%%>Meaning</th>\n"));
      f_out.WriteString(_T("</tr>\n"));
      f_out.WriteString(sMembers);
      f_out.WriteString(_T("</table><br>\n"));
   };

   HtmlDone(f_out);
   f_out.Close();
};

void ParseStruct(CStdioFile &f, 
                 LPCTSTR szPath,
                 LPCTSTR szSubPath,
                 CStdioFile &flist,
                 CString &s)
//
// Parse IDL "struct" defintions
//
{
   if( s.Find(_T("struct"))<0 ) return; // not an struct

   BfxRemoveToken(s,_T(" ")); // remove "typedef"
   tAttributes attribs;
   ParseAttributes(f, s,attribs);

   if( BfxRemoveToken(s,_T(" "))!=_T("struct") ) return; // error
   CString sName;
   sName = BfxRemoveToken(s,_T(" "));

   CString sFilename;
   CString sLowerName(sName);
   sLowerName.MakeLower();
   sFilename.Format(_T("%sstructs\\%s.htm"), szPath, (LPCTSTR)sLowerName);

   CString sPath;
   sPath.Format(_T("%sstructs\\"), szSubPath);
   AddTOC(sTOC_Structs, nTOC_Structs, sLowerName, sPath, sName);

   // Add entry to overview html file
   CString tmp;
   tmp.Format(_T("<p class=indent1><a href=\"%s.htm\"><b>%s</b></a></p>\n"),
      (LPCTSTR)sLowerName,
      (LPCTSTR)sName);
   flist.WriteString(tmp);

   // Create the new entry html file
   CStdioFile f_out;
   f_out.Open(sFilename,CFile::modeWrite|CFile::typeText|CFile::modeCreate);
   HtmlInit(f_out,sName);
   Keywords.Add(_T("Structures"));
   Keywords.Add(sName);

   if( !attribs.HelpString.IsEmpty() ) {
      tmp.Format(_T("<p>%s</p>\n"), attribs.HelpString);
      f_out.WriteString(tmp);
   };
   f_out.WriteString(Config.Tags.sCodeBegin); 
   tmp.Format(_T("<b>typedef struct %s {</b>\n"), (LPCTSTR)sName);
   f_out.WriteString(tmp);

   BOOL bHasHelp = FALSE;
   CString sMembers;

   while( FileReadString(f,s)==TRUE ) {
      // Process struct
      // Outer loop:
      s.TrimLeft();
      if( s.IsEmpty() ) continue;
      if( s.Left(2)==_T("//") ) continue;
      if( s.Left(1)==_T("}") ) break; // end of struct

      // Parse
      if( s.Left(1)==_T("[") ) {
         tAttributes attribs;
         ParseAttributes(f, s, attribs);
         //
         FileReadString(f,s);
         s.TrimLeft();
         //
         if( !attribs.HelpString.IsEmpty() ) {
            // Attribute has an "helpstring" property, which we will
            // add later in the "Members:" html section.
            int pos = s.FindOneOf(_T(" ,=["));
            if( pos>=0 ) tmp = s.Left(pos); else tmp = s;
            sMembers += _T("<tr valign=top>\n");
            sMembers += _T("<td width=45%%><b>") + tmp + _T("</b></td>\n");
            sMembers += _T("<td width=55%%>") + attribs.HelpString + _T("</td>\n");
            sMembers += _T("</tr>\n");
            bHasHelp = TRUE;
         };
      };
      int pos = s.Find(_T("//")); if( pos>=0 ) s = s.Left(pos); // remove "//" comment
      tmp.Format(_T("    %s\n"), (LPCTSTR)s);
      f_out.WriteString(tmp);
   };

   tmp.Format(_T("<b>} %s</b>"), (LPCTSTR)sName);
   f_out.WriteString(tmp);
   f_out.WriteString(Config.Tags.sCodeEnd);

   f_out.WriteString(_T("\n<h4>Members</h4>"));
   if( !bHasHelp ) {
      // no HELPSTRING for individual members
      f_out.WriteString(_T("<p>See individual functions.</p>"));
   }
   else {
      f_out.WriteString(_T("<table cellspacing=4 cols=2>\n"));
      f_out.WriteString(_T("<tr valign=top>\n"));
      f_out.WriteString(_T("<th align=left width=45%%>Value</th>\n"));
      f_out.WriteString(_T("<th align=left width=55%%>Meaning</th>\n"));
      f_out.WriteString(_T("</tr>\n"));
      f_out.WriteString(sMembers);
      f_out.WriteString(_T("</table><br>\n"));
   };

   HtmlDone(f_out);
   f_out.Close();
};

void ParseInterface(CStdioFile &f, 
                    LPCTSTR szPath,
                    LPCTSTR szSubPath,
                    CStdioFile &flist,
                    CString &s)
//
// Parse IDL "interface" defintions
//
{
   if( s.Find(_T(";"))>=0 ) return; // a forward declare

   // Primitive progress bar
   cout << ".";

   CString sClassName, sLowerClassName;
   BfxRemoveToken(s,_T(" ")); // remove "interface"
   sClassName = BfxRemoveToken(s,_T(":"));
   sClassName.TrimRight();
   sLowerClassName = sClassName;
   sLowerClassName.MakeLower();
   
   CString sPath;
   sPath.Format(_T("%sinterfaces\\%s"), szPath, (LPCTSTR)sLowerClassName);
   CDir dir;
   dir.Create(sPath);
   ADDBACKSLASH(sPath);
   
   CString sFilename;
   sFilename.Format(_T("%s%s.htm"), (LPCTSTR)sPath, (LPCTSTR)sLowerClassName);

   sPath.Format(_T("%sinterfaces\\%s\\"), szSubPath, (LPCTSTR)sLowerClassName);
   AddTOC(sTOC_Interfaces, nTOC_Interfaces, sLowerClassName, sPath, sClassName);

   FileReadString(f,s); // skip the "{" line

   // Add entry to overview html file
   CString tmp;
   tmp.Format(_T("<p class=indent1><a href=\"%s\\%s.htm\"><b>%s</b></a></p>\n"),
      (LPCTSTR)sLowerClassName,
      (LPCTSTR)sLowerClassName,
      (LPCTSTR)sClassName);
   flist.WriteString(tmp);

   // Create the new entry html file
   CStdioFile f_out;
   f_out.Open(sFilename,CFile::modeWrite|CFile::typeText|CFile::modeCreate);
   HtmlInit(f_out,sClassName,3);

   //
   // Class description
   //
   tSourceDescription SourceInfo;
   tmp.Empty();
   ParseSourceCodeDocumentation(sClassName,tmp,SourceInfo);

   if( !SourceInfo.Description.IsEmpty() ) {
      tmp.Format(_T("\n<p>%s</p>\n"), (LPCTSTR)SourceInfo.Description);
      f_out.WriteString(tmp);
   }
   else
      f_out.WriteString(_T("\n<p>&nbsp;</p>\n"));

   //
   // Parse IDL file for entries
   //
   enum { PROPERTY=0, METHOD=1 };
   for( int i=0; i<2; i++ ) {
      DWORD filepos = f.GetPosition();
      // Parse
      CString sLastID;
      long index = 0;
      while( FileReadString(f,s)==TRUE ) {
         // Process interface
         s.TrimLeft();
         if( s.IsEmpty() ) continue;
         if( s.Left(2)==_T("//") ) continue;
         if( s.Left(1)==_T("}") ) break; // end of interface
         //
         tAttributes attribs;
         ParseAttributes(f, s, attribs);
         if( attribs.IsHidden ) continue; // don't show hidden functions
         if( sLastID == attribs.sId ) continue; // skip get/set properties (only keep one)
         if( (i==PROPERTY) && !attribs.IsProperty ) continue;
         if( (i==METHOD) && attribs.IsProperty ) continue;
         //
         if( index==0 ) {
            switch( i ) {
            case PROPERTY:
               f_out.WriteString(_T("\n<h4>Properties:</h4>\n"));
               break;
            case METHOD:
               f_out.WriteString(_T("\n<h4>Methods:</h4>\n"));
               break;
            };
            f_out.WriteString(_T("<table cellspacing=4 cols=2>\n"));
         }
         
         // If method is multiline, then load rest now
         while( TRUE ) {
            s.TrimRight();
            if( s.Right(1)!=_T(",") && s.Right(1)!=_T("(")  ) break;
            FileReadString(f,tmp);
            s += tmp;
         };

         // Extract name
         CString sName, sLowerName;
         tmp = s;
         BfxRemoveToken(tmp, _T("HRESULT "));
         sName = BfxRemoveToken(tmp, _T("("));
         sName.TrimLeft();
         sName.TrimRight();
         sLowerName = sName;
         sLowerName.MakeLower();

         if( sName.IsEmpty() ) return; // some error

         f_out.WriteString(_T("<tr valign=top>\n"));
         tmp.Format(_T("<td width=45%%><a href=\"methods/%s.htm\"><b>%s</b></a></td>\n"), 
            (LPCTSTR)sLowerName, 
            (LPCTSTR)sName);
         f_out.WriteString(tmp);
         if( !attribs.HelpString.IsEmpty() )
            tmp.Format(_T("<td width=55%%>%s</td>\n"), (LPCTSTR)attribs.HelpString);
         else
            tmp = _T("<td width=55%%></td>\n");
         f_out.WriteString(tmp);
         f_out.WriteString(_T("</tr>\n"));

         ParseFunc(f,szPath,sClassName,sName,attribs,s);

         sLastID = attribs.sId;
         index++;
      };
      if( index>0 ) f_out.WriteString(_T("</table><br>\n"));

      // For the first pass, we must spool back to the start of the methods
      // because we wish to parse again
      if( i==PROPERTY ) {
         f.Seek(filepos,CFile::begin);
      };
   };

   Keywords.Add(sClassName);
   Keywords.Add(sClassName + _T(" class members"));
   Keywords.Add(_T("Interfaces"));
   HtmlDone(f_out);
   f_out.Close();
};


/////////////////////////////////////////////////////////////////////////////
// 
// Main loop which scans the IDL file for this we wanna
// add to the html help document
//

void Start(LPCTSTR szPath, LPCTSTR szSubPath, LPCTSTR szInputFilename)
{
   CStdioFile f;
   if( f.Open( szInputFilename, CFile::typeText|CFile::modeRead )==FALSE ) {
      cerr << endl << "Error: Cannot open IDL file." << endl;
      return;
   };

   //
   // Create directories
   //
   CString sPath(szPath);
   CString sSubPath(szSubPath);
   CDir dir;
   dir.Create(sPath + _T("interfaces"));
   dir.Create(sPath + _T("constants"));
   dir.Create(sPath + _T("structs"));
   cout << "..."; // simple progress bar

   CString sFilename;

   //
   // Open various all.html list files
   //
   CStdioFile f_list;
   sFilename = sPath + _T("interfaces\\all.htm");
   f_list.Open(sFilename,CFile::modeWrite|CFile::typeText|CFile::modeCreate);
   HtmlInit(f_list,_T("Interfaces"));
  
   CStdioFile f_const;
   sFilename = sPath + _T("constants\\all.htm");
   f_const.Open(sFilename,CFile::modeWrite|CFile::typeText|CFile::modeCreate);
   HtmlInit(f_const,_T("Enumerated Types"));

   CStdioFile f_struct;
   sFilename = sPath + _T("structs\\all.htm");
   f_struct.Open(sFilename,CFile::modeWrite|CFile::typeText|CFile::modeCreate);
   HtmlInit(f_struct,_T("Structures"));

   InitTOC(szSubPath);

   //
   // Parse IDL file
   //
   CString s;
   BOOL bHidden;
   while( FileReadString(f,s)==TRUE ) {
      // Process IDL file
      // Outer loop:
      s.TrimLeft();
      if( s.IsEmpty() ) continue;
      if( s.Left(2)==_T("//") ) continue;
      //
      CString tmp(s);
      tmp.MakeLower();
      if( tmp.Left(1)==_T("[") ) bHidden=FALSE;
      if( tmp.Find(_T("hidden"))>=0 ) bHidden=TRUE;
      if( tmp.Left(7)==_T("typedef") ) {
         if( tmp.Find(_T("enum"))>0 )
            ParseEnum(f,sPath,sSubPath,f_const,s);
         else if( tmp.Find(_T("struct"))>0 )
            ParseStruct(f,sPath,sSubPath,f_struct,s);
      };
      if( tmp.Left(9)==_T("interface") )
         if( !bHidden ) ParseInterface(f,sPath,sSubPath,f_list,s);
   };

   //
   // Done
   //
   HtmlDone(f_list);
   f_list.Close();
   HtmlDone(f_const);
   f_const.Close();
   HtmlDone(f_struct);
   f_struct.Close();

   DoneTOC();
   CreateTOC(sPath);

   f.Close();
};


/////////////////////////////////////////////////////////////////////////////
// The one and only application object

void Init()
{
   LPCTSTR szSection = _T("settings");
   CString s, tmp;
   int i;

   Config.sTextHeader = ini.GetString(szSection,_T("HeaderText"));
   Config.sTextFooter = ini.GetString(szSection,_T("FooterText"));
   Config.sFileExtension = ini.GetString(szSection,_T("FileExtension"),_T(".CPP"));
   Config.sStyleSheet = ini.GetString(szSection,_T("StyleSheet"));
   Config.sTOCFilename = ini.GetString(szSection,_T("ContentsFile"), _T("template.hhc"));
   Config.sIgnoreLinks = ini.GetString(szSection,_T("IgnoreLinks"));
   Config.sIgnoreLinks += _T(",IUnknown,IDispatch,");

   // Set up Search Path
   Config.SearchPaths.RemoveAll();
   Config.SearchPaths.Add(BfxGetAppPath());
   tmp = ini.GetString(szSection,_T("IncludePath"));
   while( !tmp.IsEmpty() ) {
      CString sPath = BfxRemoveToken(tmp,_T(';'));
      ADDBACKSLASH(sPath);
      Config.SearchPaths.Add( sPath );
   };

   // Set up default texts
   i = 0;
   tmp = ini.GetString(szSection,_T("DefaultTexts"));
   while( !tmp.IsEmpty() ) {
      CString sTitle;
      CString sName = BfxRemoveToken(tmp,_T(','));
      sName.TrimLeft();
      sName.TrimRight();
      sTitle.Format(_T("Default Text - %s"), (LPCTSTR)sName);

      Config.Defaults[i].Name = sName;
      Config.Defaults[i].Default = ini.GetString(sTitle,_T("Default"));
      if( Config.Defaults[i].Default.IsEmpty() ) break;

      int j=1;
      do {
         CString sFilter;
         sFilter.Format(_T("Filter%d"),j);
         s = ini.GetString(sTitle,sFilter);
         if( !s.IsEmpty() ) {
            CString tmp = BfxRemoveToken(s,_T(','));
            Config.Defaults[i].FilterNames.Add(tmp);
            Config.Defaults[i].FilterValues.Add(s);
         };
         j++;
      } while( !s.IsEmpty() );
      // Ready for next
      i++;
   };
   Config.nDefaults = i+1;

   // Get all error code strings
   CString sSection;
   sSection = _T("default text - Error Strings");
   Config.ErrorStrings.sS_OK = ini.GetString(sSection,_T("S_OK"), _T("The operation was successfull."));
   Config.ErrorStrings.sE_POINTER = ini.GetString(sSection,_T("E_POINTER"), _T("A NULL pointer was supplied as an argument."));
   Config.ErrorStrings.sE_INVALIDARG = ini.GetString(sSection,_T("E_INVALIDARG"), _T("An invalid argument argument was passed."));
   Config.ErrorStrings.sE_UNEXPECTED = ini.GetString(sSection,_T("E_UNEXPECTED"), _T("An unexpected error occoured."));
   Config.ErrorStrings.sE_FAIL = ini.GetString(sSection,_T("E_FAIL"), _T("A general error occoured."));

   // Get some html tags
   sSection = _T("Html Tags");
   Config.Tags.sBodyBegin.Format(_T("%s\n"), ini.GetString(sSection,_T("BodyBegin"), _T("<body>")));
   Config.Tags.sBodyEnd.Format(_T("%s\n"), ini.GetString(sSection,_T("BodyEnd"), _T("</body>")));
   Config.Tags.sCodeBegin.Format(_T("\n%s\n"), ini.GetString(sSection,_T("CodeBegin"), _T("<br><table bgcolor=""#EEEEEE"" width=""100%""><tr><td><pre><code>")));
   Config.Tags.sCodeEnd.Format(_T("%s\n"), ini.GetString(sSection,_T("CodeEnd"), _T("</code></pre></td></tr></table>")));
   Config.Tags.sTitleBegin.Format(_T("\n%s"), ini.GetString(sSection,_T("TitleBegin"), _T("<h1>")));
   Config.Tags.sTitleEnd.Format(_T("%s\n"), ini.GetString(sSection,_T("TitleEnd"), _T("</h1>")));

}

int _tmain(int argc, TCHAR* argv[], TCHAR* envp[])
{
   // initialize MFC and print and error on failure
   if (!AfxWinInit(::GetModuleHandle(NULL), NULL, ::GetCommandLine(), 0))
   {
      cerr << _T("Fatal Error: MFC initialization failed") << endl;
      return 1;
   };

   if( argc<2 ) {
      cerr << _T("Error: Invalid number of arguments.") << endl;
      cout << _T("Usage: idlhelp <.idl-file> [<subpath>] [<extension>]") << endl;
      return 2;
   };
   
   CString sPath;
   CString sFilename;
   sFilename = argv[1];
   CDir::ExpandFilename(sFilename);

   sPath = BfxGetAppPath();   
   ini.SetIniFilename( BfxGetAppPath() + AfxGetAppName() + _T(".INI") );

   Init();

   // Get subpath argument
   CString sSubPath;
   if( argc>=3 ) {
      sSubPath = argv[2];
      sPath += sSubPath;
      ADDBACKSLASH(sPath);
      CDir::ExpandPath(sPath);
      if( sPath.GetLength() < 4 ) {
         cerr << _T("Error: Invalid path.") << endl;
         cout << _T("Usage: idlhelp <.idl-file> [<subpath>] [<extension>]") << endl;
         return 3;
      };
      CDir dir;
      dir.Delete(sPath);
      dir.Create(sPath);
   };
   ADDBACKSLASH(sSubPath);

   // Get file extension argument
   if( argc>=4 ) {
      Config.sFileExtension = argv[3];
      if( Config.sFileExtension.IsEmpty() )
         Config.sFileExtension = _T(".CPP");
      if( Config.sFileExtension[0] != _T('.') )
         Config.sFileExtension = _T(".CPP");
   };

   // Go
   cout << _T("Processing.");
   Start(sPath,sSubPath,sFilename);
   cout << endl << _T("Done.") << endl;

   return 0;
}
